package org.pathwaycommons.trans;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Iterator;
import java.util.TreeMap;

import org.genemania.dw.util.GenUtil;


/**
 * Encapsulates the CV mapping functionality for a translation. It carries the
 * 'bean structure' plus additional functions. Currently
 * implements the CV mapping for the BIND to PSI-MI 2.5 translation.
 * additional info list is a list of att:value pairs.
 * The CV mapping file is a tab delimited flat file with the ordered columns:
 * CV Type, Special Handling, Source Term, Source ID, Target Term, Target ID,
 * followed by zero to many additional info columns.
 * The 'Source Term' is assumed as the unique key in an entry.
 * The PSI-MI target term can be of the format: ShortLabel:FullName or
 * generic/unknown term: source term (repeated/standardized).
 * The special handling bit indicates that this class carries additional
 * 'business rules' for the mapping of that CV (e.g. setting the PSI-MI X-ref
 * type for the BIND term 'WormBase' for a protein).
 * 'N/A' used for missing entries in the file, but not saved at the object
 *  level.
 * Target PSI-MI term format: shortLabel;fullName (if both available)
 *                            fullName (if not shortLabel)
 *              generic (e.g. 'participant xref');uniform source term (repeated)
 * In the last case, the targetID is ignored, as non is really suitable. 
 *
 * @author rashadbadrawi
 */

public class CVMapEntry {

    protected static TreeMap <String, CVMapEntry> CVMap =
                                          new TreeMap <String, CVMapEntry> ();

    //required padding for the PSIMI IDs
    private static final int PSIMI_PAD_LEN = 4;
    private static final String PSIMI_PAD_STR = "0";

    private String targetCategory;
    private String sourceName;
    private String targetName;
    private String sourceTerm;
    private String targetTerm;
    private String sourceID;
    private String targetID;
    private boolean isSpecialCase;
    private TreeMap <String, String> addInfoMap = new TreeMap <String, String> ();

    protected CVMapEntry () {}

    protected void setSourceName (String sourceName) {

        GenUtil.validateString(sourceName);
        this.sourceName = sourceName;
    }

    protected void setTargetName (String targetName) {

        GenUtil.validateString (targetName);
        this.targetName = targetName;
    }

    protected void setTargetCategory (String category) {

        GenUtil.validateString(category);
        this.targetCategory = category;
    }

    protected void setSourceTerm (String sourceTerm) {

        GenUtil.validateString(sourceTerm);
        this.sourceTerm = sourceTerm;
    }

    protected void setTargetTerm (String targetTerm) {

        GenUtil.validateString(targetTerm);
        if (!GenUtil.NA.equals (targetTerm)) {
            this.targetTerm = targetTerm;
        }
    }

    protected void setSourceID (String sourceID) {

        GenUtil.validateString(sourceID);
        if (!GenUtil.NA.equals (sourceID)) {
            this.sourceID = sourceID;
        }
    }

    protected void setTargetID (String targetID) {

        GenUtil.validateString(targetID);
        if (!GenUtil.NA.equals (targetID)) {
            if (PSIMITags.DB_PSIMI.equals (targetName)) {
                for (int i = targetID.length (); i < PSIMI_PAD_LEN; i++) {
                    targetID = PSIMI_PAD_STR + targetID;
                }
                targetID = PSIMITags.MI_PREFIX + targetID;
            }
            this.targetID = targetID;
        }
    }

    protected void setIsSpecialCase (boolean isSpecialCase) {

        this.isSpecialCase = isSpecialCase;
    }

    protected void setAddInfo (TreeMap <String, String> addInfoMap) {

        GenUtil.validateNotNull(addInfoMap);
        this.addInfoMap = addInfoMap;
    }

    protected String getTargetCategory () { return targetCategory; }

    protected String getSourceName () { return sourceName; }

    protected String getTargetName () { return targetName; }

    protected String getSourceTerm () { return sourceTerm; }

    protected String getTargetTerm () { return targetTerm; }

    protected String getSourceID () { return sourceID; }

    protected String getTargetID () { return targetID; }

    protected boolean isSpecialCase () { return isSpecialCase; }

    protected String getPSIMITerm_SL () {

        return getPSIMITerm (PSIMITags.SHORT_LABEL);
    }

    protected String getPSIMITerm_FL () {

        return getPSIMITerm (PSIMITags.FULLNAME);
    }

    //helper method
    protected String getPSIMITermID () {

        if (!getTargetTerm ().contains (GenUtil.SEMICOLON)) {
            return getTargetID();
        } else {
            String [] tempArr = getTargetTerm ().split (GenUtil.SEMICOLON);
            if (tempArr [0].trim ().equals (PSIMITags.GEN_XREF)  ||
                tempArr [0].trim().equals (PSIMITags.TYPE_UP_SL) ||
                (tempArr [0].trim().equals(PSIMITags.GEN_IDM) &&
                 !tempArr [1].trim ().equals (PSIMITags.GEN_IDM_FN))) {
                return null;
            } else {
                return getTargetID ();
            }
        }
    }
    //helper method
    private String getPSIMITerm (String termType) {

        if (!getTargetTerm ().contains (GenUtil.SEMICOLON)) {
            return getTargetTerm ();
        }
        String [] tempArr = getTargetTerm ().split (GenUtil.SEMICOLON);
        if (tempArr [0].trim ().equals (PSIMITags.GEN_XREF)  ||
            tempArr [0].trim().equals (PSIMITags.TYPE_UP_SL) ||
            (tempArr [0].trim().equals(PSIMITags.GEN_IDM) &&
             !tempArr [1].trim ().equals (PSIMITags.GEN_IDM_FN))) {
            return tempArr [1].trim ();
        } else {
            if (PSIMITags.SHORT_LABEL.equals (termType)) {
                return tempArr [0].trim ();
            } else if (PSIMITags.FULLNAME.equals(termType)) {
                return tempArr [1].trim();
            }
        }

        return null;
    }

    protected String getAddInfo (String attName, String val) {

        GenUtil.validateString(attName);
        if (!isSpecialCase()) {
           return addInfoMap.get (attName);
        } else {
           return handleSpecialCase (attName, val);
        }
    }

    private String handleSpecialCase (String attName, String val) {

        if (getSourceName ().equals (PSIMITags.BIND_SL)) {
            if (getSourceTerm ().equalsIgnoreCase(BINDTags.SC_WB)) {
                return handleSpecialCase_BIND_WB (attName, val);
            }
        }

        return null;
    }

    private String handleSpecialCase_BIND_WB (String attName, String val) {

        GenUtil.validateString(val);
        //System.out.println ("Processing CV - special case: " + attName + " " + val);
        if (PSIMITags.TYPE_PROTEIN_SL.equalsIgnoreCase(attName)) {
            if (val.startsWith (BINDTags.SC_WB_PAT_1)) {
                return PSIMITags.REFTYPE_GP;
            } else if (val.startsWith (BINDTags.SC_WB_PAT_2)) {
                return PSIMITags.REFTYPE_IDENT;
            } else if (val.startsWith (BINDTags.SC_WB_PAT_3)) {
                return PSIMITags.REFTYPE_IDENT;
            } else { //like ZK858.4 or ced-13 or one odd case 'WormBase: T24D3.1'
                return PSIMITags.REFTYPE_GP;         
            }
        } else if (PSIMITags.TYPE_DNA_SL.equalsIgnoreCase(attName)) {
            if (val.startsWith (BINDTags.SC_WB_PAT_1)) {
                return PSIMITags.REFTYPE_IDENT;
            } else {
                return PSIMITags.REFTYPE_IDENT;         
            }
        } else if (PSIMITags.TYPE_RNA_SL.equalsIgnoreCase(attName)) {
            if (val.startsWith (BINDTags.SC_WB_PAT_1)) {
                return PSIMITags.REFTYPE_GP;
            } else {
                return PSIMITags.REFTYPE_GP;       
            }
        } else if (PSIMITags.TYPE_GENE_SL.equalsIgnoreCase(attName)) {
            if (val.startsWith (BINDTags.SC_WB_PAT_2)) {
                return PSIMITags.REFTYPE_GP;                             //?
            } 
        }

        return null;
    }

    protected static void loadCVMappings (String CVMappingFileName, String sourceName,
                                     String targetName) {

        GenUtil.validateString(CVMappingFileName);
        GenUtil.validateString(sourceName);
        GenUtil.validateString(targetName);
        if (sourceName.equals (PSIMITags.BIND_SL)) {
            loadBINDCVMappings (CVMappingFileName, sourceName, targetName);
        } else {
            throw new IllegalArgumentException ("Invalid mapping source: " +
                      sourceName);
        }
    }

    private static void loadBINDCVMappings (String CVMappingFileName, String
                                         sourceName, String targetName) {

        GenUtil.validateString(CVMappingFileName);

        try {
            BufferedReader br = new BufferedReader
                                    (new FileReader (CVMappingFileName));
            String line;
            while ((line = br.readLine ()) != null) {
                String tempArr [] = line.split (GenUtil.TAB);
                CVMapEntry mapEntry = new CVMapEntry ();
                mapEntry.setSourceName (sourceName);
                mapEntry.setTargetName (targetName);
                mapEntry.setTargetCategory (tempArr [0]);
                mapEntry.setIsSpecialCase(Boolean.parseBoolean(tempArr [1]));
                mapEntry.setSourceTerm(tempArr [2]);
                mapEntry.setSourceID (tempArr [3]);
                mapEntry.setTargetTerm (tempArr [4]);
                mapEntry.setTargetID (tempArr [5]);
                if (tempArr [0].equals (PSIMITags.XREF)) {
                    TreeMap <String, String> addInfoMap = new TreeMap <String, String> ();
                    addInfoMap.put (PSIMITags.TYPE_PROTEIN_SL, tempArr[6]);
                    addInfoMap.put (PSIMITags.TYPE_DNA_SL, tempArr[7]);
                    addInfoMap.put (PSIMITags.TYPE_RNA_SL, tempArr[8]);
                    addInfoMap.put (PSIMITags.TYPE_GENE_SL, tempArr[9]);
                    addInfoMap.put (PSIMITags.TYPE_SM_SL, tempArr[10]);
                    addInfoMap.put (PSIMITags.TYPE_COMPLEX_SL, tempArr[11]);
                    addInfoMap.put (PSIMITags.TYPE_UP_SL, tempArr[12]);
                    mapEntry.setAddInfo (addInfoMap);
                }
                if (CVMap.containsKey(mapEntry.getSourceTerm())) {
                    System.err.println ("Error: non unique source term: " +
                                        mapEntry.getSourceTerm ());
                } //overwrite, anyway.
                CVMapEntry.CVMap.put(mapEntry.getSourceTerm(), mapEntry);
                //System.out.println (mapEntry.toString ());
            }
            System.out.println ("Done reading CV Mapping from file: " + CVMappingFileName);
        } catch (IOException ioe) {
            ioe.printStackTrace ();
            System.err.println ("Unable to load mapping CVs from file: " +
                                 CVMappingFileName);
        }
    }

    protected static CVMapEntry getMappedEntry (String sourceTerm) {

        GenUtil.validateString(sourceTerm);
        
        return CVMapEntry.CVMap.get (sourceTerm);
    }

    @Override
    public String toString () {

        String tempStr = getTargetCategory () + GenUtil.TAB;
        tempStr += getSourceName() + GenUtil.TAB;
        tempStr += getSourceID () + GenUtil.TAB;
        tempStr += getSourceTerm() + GenUtil.TAB;
        tempStr += getTargetName () + GenUtil.TAB;
        tempStr += getTargetID () + GenUtil.TAB;
        tempStr += getTargetTerm () + GenUtil.TAB;
        Iterator iterator = addInfoMap.keySet().iterator();
        while (iterator.hasNext ()) {
            String att = (String)iterator.next();
            String val = addInfoMap.get(att);
            tempStr += att + "(" + val + ")" + GenUtil.SEMICOLON;
        }

        return tempStr;
    }
}
