package org.pathwaycommons.trans;

import java.io.File;
import java.io.PrintWriter;
import java.util.ArrayList;

import org.genemania.dw.util.DefParams;
import org.genemania.dw.util.GenUtil;

/**
 * Abstract, type control for all PSI-MI translators.
 * Semi-factory for the translator.
 *
 * @author rashadbadrawi
 */

public abstract class PSIMITrans implements Trans {

    public static final String BIND_TO_PSIMI25 = "BIND_TO_PSIMI25";
    protected static final String PSIMI_FILE_SUFFIX = "_PSIMI25";
    protected static final String NO_VAL = "NO_VALUE";
    protected static final String BAD_ENTRIES = "BAD_ENTRIES.txt";
    protected static PrintWriter log = GenUtil.getDefaultLog();

    private static ArrayList <String> translationTypesList;

    static {
        translationTypesList = new ArrayList <String> ();
        translationTypesList.add (PSIMITrans.BIND_TO_PSIMI25);
    }

    protected PSIMITrans () {}
    
    public abstract void translate () throws Exception;

    public static PSIMITrans translatorFactory (String translationType,
                  String inputFilesPath)  throws Exception {

        if (!isSupportedTranslation (translationType)) {
            throw new IllegalArgumentException ("Unsupported translation type: " +
                                                translationType);
        }
        GenUtil.validateString(inputFilesPath);
        if (!new File (inputFilesPath).exists ()) {
            throw new IllegalArgumentException ("Invalid input file(s) path: " +
                                               inputFilesPath);
        }
        if (translationType.equals (PSIMITrans.BIND_TO_PSIMI25)) {
            return new BINDToPSIMITrans (inputFilesPath);
        }

        return null;
    }

    public static boolean isSupportedTranslation (String translationType) {

        return PSIMITrans.translationTypesList.contains(translationType);
    }

    public static void main(String args[]) {

        log = GenUtil.getDefaultLog();
        log.println(GenUtil.getTimeStamp());
        GenUtil.registerStart();
        String usageMsg = "Usage: PSIMITranslator TranslationType [" +
                PSIMITrans.BIND_TO_PSIMI25 + "]";
        String warnMsg = "WARNING: Missing command line args, using defaults";
        if (args == null || args.length < 2) {
            log.println(warnMsg);
            log.println(usageMsg);
            System.out.println(warnMsg);
            System.out.println(usageMsg);
            args = DefParams.getCommandLineArgs(PSIMITrans.class.getName());
        }
        try {
            PSIMITrans BINDTrans = PSIMITrans.translatorFactory (args [0], args [1]);
            BINDTrans.translate();
        } catch (Throwable e) {
            e.printStackTrace();
            e.printStackTrace(log);
        } finally {
            log.println(GenUtil.getExecTimeStr());
            log.flush();
            log.close();
        }
    }
}
