package org.pathwaycommons.trans;

import java.io.BufferedWriter;
import java.io.IOException;
import java.util.ArrayList;

import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.Namespace;

import org.pathwaycommons.trans.util.XMLUtil;
import org.genemania.dw.util.GenUtil;

/**
 * BIND To PSI-MI translation for complexes. Modelled as a subclass to the main
 * BIND translator.
 *
 * @author rashadbadrawi
 */

public class BINDToPSIMITransComplex extends BINDToPSIMITrans {

    private int cntTotalComplex, cntTransComplex, cntSkippedComplex,
                cntBadComplex;
    private String BINDComplexID;                   //for easier handling

    //overlooking the use of 'super'
    protected BINDToPSIMITransComplex (String inputFilesPath, String fileName, Namespace psins,
            int interactionID, int interactorID, int partID, int expID, BufferedWriter bw) {

        super ();
        this.psins = psins;
        this.fileName = fileName;
        this.inputFilesPath = inputFilesPath;
        this.bw = bw;
        this.interactionID = interactionID;
        this.interactorID = interactorID;
        this.partID = partID;
        this.expID = expID;
    }

    protected void translateAllComplexes (Element entryE, Element BINDSubmitE)
                                       throws JDOMException, IOException {

        Element interactionListE = entryE.getChild(PSIMITags.INT_LIST, this.psins);
        //if no interactions in this file/species
        if (interactionListE == null) {
            interactionListE = new Element (PSIMITags.INT_LIST, this.psins);
            entryE.addContent (interactionListE);
        }
        ArrayList <Element> bindIntList = XMLUtil.getElement (BINDTags.COMPLEX,
                    BINDSubmitE, 1);
        for (int i = 0; i < bindIntList.size (); i++) {
            translateComplex (interactionListE, bindIntList.get (i));
        }
        System.out.println ("For file: " + getInputFilesPath() + ": ");
        System.out.println ("Total Complexes: " + this.cntTotalComplex +
                            " Translated: " + this.cntTransComplex +
                            " Skipped: " + this.cntSkippedComplex +
                            " Bad: " + this.cntBadComplex);
    }

    private void translateComplex (Element interactionListE, Element BINDCompE)
                                  throws JDOMException, IOException {

        String complexDesc = NO_VAL, complexNum = NO_VAL, orderedFlag = NO_VAL;
        BINDComplexID = NO_VAL;
        ArrayList <String> interactionIDsList = new ArrayList <String> ();

        //get complex ID
        ArrayList <Element> tempList = XMLUtil.getElement (BINDTags.COMPLEX_ID,
             BINDCompE.getChild(BINDTags.COMPLEX_MCID),
             XMLUtil.getDepth (BINDCompE.getChild(BINDTags.COMPLEX_MCID).getName()));
        if (XMLUtil.validateSingle(BINDTags.COMPLEX_ID, tempList)) {
            BINDComplexID = getDefText(tempList.get (0), NO_VAL);
        }
        if (BINDComplexID.equals (NO_VAL)) {
            System.err.println ("Error 1: NO ID for complex.");
            this.badInteractionData = true;
        }

        //get complex description
        tempList = XMLUtil.getElement (BINDTags.COMPLEX_DESC, BINDCompE,
                                       XMLUtil.getDepth (BINDCompE.getName()));
        if (XMLUtil.validateSingle(BINDTags.COMPLEX_DESC, tempList)) {
            complexDesc = getDefText(tempList.get (0), NO_VAL);
        }
        if (complexDesc.equals (NO_VAL)) {
            System.err.println ("Error 2: NO Desc for complex: " + BINDComplexID);
            this.badInteractionData = true;
        }

        //Total num of subunits. There is another 'mol_obj' subnumber that is never used.
        tempList = XMLUtil.getElement (BINDTags.COMPLEX_NUM, BINDCompE,
                                       XMLUtil.getDepth (BINDCompE.getName()));
        if (XMLUtil.validateSingle(BINDTags.COMPLEX_NUM, tempList)) {
            complexNum = getDefText(tempList.get (0), NO_VAL);
        }
        if (complexNum.equals (NO_VAL)) {
            System.err.println ("Warning: NO count for complex: " + BINDComplexID);
        }

        //get interaction IDs from the interactions list
        tempList = XMLUtil.getElement (BINDTags.COMPLEX_INT_LIST, BINDCompE,
                                       XMLUtil.getDepth (BINDCompE.getName()));
        if (!XMLUtil.validateSingle(BINDTags.COMPLEX_INT_LIST, tempList)) {
            System.err.println ("Warning: No interaction list found for complex: " + BINDComplexID);
        } else {
            ArrayList <Element> interactionIDList;
            interactionIDList = XMLUtil.getElement (BINDTags.INTERACTION_ID, tempList.get (0),
                                           XMLUtil.getDepth (tempList.get (0).getName()));
            if (XMLUtil.validateExists(BINDTags.INTERACTION_ID, interactionIDList)) {
                for (int i = 0; i < interactionIDList.size (); i++) {
                    if (getDefText(interactionIDList.get (i), NO_VAL).equals (NO_VAL)) {
                        System.err.println ("Error 3: Bad interaction ID for complex: " + BINDComplexID);
                        continue;
                    }
                    interactionIDsList.add (interactionIDList.get (i).getTextTrim());
                }
            }
        }

        //complex order flag
        tempList = XMLUtil.getElement (BINDTags.COMPLEX_ORDERED_FLAG, BINDCompE,
                                       XMLUtil.getDepth (BINDCompE.getName()));
        if (XMLUtil.validateSingle(BINDTags.COMPLEX_ORDERED_FLAG, tempList)) {
            orderedFlag = tempList.get (0).getAttributeValue(BINDTags.VALUE_ATT);
        }
        
        //build the PSI-MI complex
        Element interactionE = new Element (PSIMITags.INT, this.psins);
        addComplex (interactionE, BINDCompE, BINDComplexID, complexDesc,
                   complexNum, orderedFlag, interactionIDsList);

        //get/add the complex division type
        tempList = XMLUtil.getElement (BINDTags.INTERACTION_COL, BINDCompE,
                            XMLUtil.getDepth (BINDCompE.getName()));
        if (XMLUtil.validateSingle (BINDTags.INTERACTION_COL, tempList)) {
            String BINDDivision = tempList.get (0).getTextTrim ();
            addAttribute (interactionE, PSIMITags.ATT_BIND_DIVISION,
                          BINDDivision);
        }
        interactionsMap.put (BINDComplexID, 
                             interactionE.getAttributeValue(PSIMITags.ID_ATT));
        //filter bad data
        if (!this.badInteractionData) {
            interactionListE.addContent (interactionE);
        } else {
            this.badInteractionData = false;
            addAttribute (interactionE, PSIMITags.ATT_BAD_DATA_FILE, this.fileName);
            XMLUtil.dump(interactionE, this.bw);
            this.bw.newLine();
            this.cntBadComplex++;
            badInteractionsMap.put (BINDComplexID,
                                    interactionE.getAttributeValue(PSIMITags.ID_ATT));
        }
        this.cntTotalComplex++;
        this.cntTransComplex++;
    }

    //interaction counter increments regardless.
    private void addComplex (Element interactionE, Element BINDCompE,
         String BINDComplexID, String shortLabel, String subunitNum,
         String orderedFlag, ArrayList <String> interactionIDsList) throws JDOMException {

        interactionE.setAttribute(PSIMITags.ID_ATT,
                              Integer.toString (this.interactionID++));
        if (!shortLabel.equals (NO_VAL)) {
            setNames (interactionE, shortLabel, shortLabel);
        } else {
            setNames (interactionE, shortLabel, null);
        }
        setCVRef (interactionE, PSIMITags.BIND_SL, PSIMITags.REFTYPE_IDENT,
                  PSIMITags.PREF, BINDComplexID);

        Element expListE = new Element (PSIMITags.EXP_LIST, this.psins);
        interactionE.addContent (expListE);
        translateExpDesc (expListE, BINDCompE, BINDComplexID);
        Element partListE = new Element (PSIMITags.PARTLIST, this.psins);
        interactionE.addContent (partListE);
        //get details of each subunit
        translateMolObj (partListE, BINDCompE, BINDComplexID, interactionIDsList);

        String intIDStr = "";
        String BINDIntIDStr = "";
        for (int i = 0; i < interactionIDsList.size (); i++) {
            BINDIntIDStr += interactionIDsList.get (i);
            intIDStr += matchIntID (interactionIDsList.get (i));
            if (i < interactionIDsList.size () - 1) {
                BINDIntIDStr += GenUtil.SEMICOLON;
                intIDStr += GenUtil.SEMICOLON;
            }
        }
        addAttribute (interactionE, PSIMITags.ATT_COMPLEX_INT_LIST, intIDStr);
        addAttribute (interactionE, PSIMITags.ATT_COMPLEX_INT_LIST_BIND, BINDIntIDStr);
        addAttribute (interactionE, PSIMITags.ATT_COMPLEX_INT_ORDERED, orderedFlag);
        addAttribute (interactionE, PSIMITags.ATT_COMPLEX_SUBUNIT_CNT, subunitNum);
    }

    //scaled down version for the no-exp cond. BIND interaction.
    private void translateExpDesc (Element expListE, Element BINDCompE, String
                                   BINDComplexID) throws JDOMException {

        Element expDescE = new Element (PSIMITags.EXP_DESC, this.psins);
        expDescE.setAttribute(PSIMITags.ID_ATT, Integer.toString(this.expID++));
        expListE.addContent (expDescE);
        Element intDetMethodE = new Element (PSIMITags.INT_DET_METHOD, this.psins);
        setNames(intDetMethodE, PSIMITags.SL_METHOD_NS, PSIMITags.SL_METHOD_NS);
        setXRef(intDetMethodE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                PSIMITags.ID_METHOD_NS, null, null);
        //get the pubmed IDs, if any
        translatePubMed (expDescE, BINDCompE);         //order is signifcant
        translatePubMed2 (expDescE, BINDCompE);
        if (expDescE.getChildren(PSIMITags.BIB, this.psins).size() == 0) {
            Element bibRefE = new Element (PSIMITags.BIB, this.psins);
            CVMapEntry CVMapped = CVMapEntry.getMappedEntry(PSIMITags.BIND_SL);
            setXRef (bibRefE, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                     CVMapped.getPSIMITermID(), BINDComplexID,
                     null, null);
            expDescE.addContent(bibRefE);
        }
        expDescE.addContent (intDetMethodE);
    }

    private void translateMolObj (Element partListE, Element BINDCompE, String
          BINDComplexID, ArrayList <String> interactionIDsList) throws JDOMException {

        ArrayList <Element> bindMolObjList = XMLUtil.getElement (BINDTags.COMPLEX_MOL_OBJ,
                            BINDCompE, XMLUtil.getDepth (BINDCompE.getName()));
        String subUnitNum;
        String BINDIntID, newIntID;
        String interactantAtt;
        Element interactionRefE;
        ArrayList <Element> tempList;
        for (int i = 0; i < bindMolObjList.size (); i++) {
            subUnitNum = NO_VAL;
            interactionRefE = null;
            BINDIntID = null;
            newIntID = null;
            interactantAtt = null;
            //get the subunit number
            tempList = XMLUtil.getElement (
                           BINDTags.COMPLEX_MOLOBJ_ID, bindMolObjList.get (i),
                           XMLUtil.getDepth (bindMolObjList.get (i).getName()));
            if (XMLUtil.validateSingle(BINDTags.COMPLEX_MOLOBJ_ID, tempList)) {
                //zero is a valid complex subunit num
                subUnitNum = tempList.get (0).getTextTrim();
            }
            if (subUnitNum.equals (NO_VAL)) {
                System.err.println ("Error 4: NO ID for subunit for complex: " + BINDComplexID);
            }
            Element partE = new Element (PSIMITags.PART, this.psins);
            partE.setAttribute(PSIMITags.ID_ATT, Integer.toString (this.partID++));
            partListE.addContent (partE);
            //track source for subunit - if from interactant A
            tempList = XMLUtil.getElement (BINDTags.COMPLEX_MOL_OBJ_SRC_A,
                    bindMolObjList.get (i), XMLUtil.getDepth (bindMolObjList.get (i).getName()));
            if (XMLUtil.validateSingle(BINDTags.COMPLEX_MOL_OBJ_SRC_A, tempList)) {
                interactionRefE = new Element (PSIMITags.INT_REF, this.psins);
                BINDIntID = tempList.get (0).getChildText(BINDTags.INTERACTION_ID).trim();
                if (!interactionIDsList.contains (BINDIntID)) {
                    System.err.println ("Error 5: Wrong interaction ID referenced in complex: " +
                            BINDIntID);
                    this.badInteractionData = true;
                }
                newIntID = matchIntID (BINDIntID);
                interactionRefE.setText(newIntID);
                partE.addContent (interactionRefE);
                interactantAtt = PSIMITags.ATT_COMPLEX_INT_REF_A;
            } else {
                //track source for subunit - if from interactant B
                tempList = XMLUtil.getElement (BINDTags.COMPLEX_MOL_OBJ_SRC_B,
                        bindMolObjList.get (i), XMLUtil.getDepth (bindMolObjList.get (i).getName()));
                if (XMLUtil.validateSingle(BINDTags.COMPLEX_MOL_OBJ_SRC_B, tempList)) {
                    interactionRefE = new Element (PSIMITags.INT_REF, this.psins);
                    BINDIntID = tempList.get (0).getChildText(BINDTags.INTERACTION_ID).trim();
                    if (!interactionIDsList.contains (BINDIntID)) {
                        System.err.println ("Error 5: Wrong interaction ID referenced in complex: " +
                                BINDIntID);
                        this.badInteractionData = true;
                    }
                    newIntID = matchIntID (BINDIntID);
                    interactionRefE.setText(newIntID);
                    partE.addContent (interactionRefE);
                    interactantAtt = PSIMITags.ATT_COMPLEX_INT_REF_B;
                } else {
                    //more if action-result is supported for interactions.
                }
            }
            //add interactor details if no reference to interaction available
            Element interactorE = new Element (PSIMITags.INTERACTOR, this.psins);
            translateInteractor (interactorE, bindMolObjList.get (i), this.interactorID++);
            if (interactantAtt == null) {
                partE.addContent (interactorE);
            } else {
                addAttribute (partE, PSIMITags.ATT_COMPLEX_INT_SRC, interactantAtt);
                addAttribute (partE, PSIMITags.ATT_COMPLEX_INT_REF_BIND, BINDIntID);
            }
            addAttribute (partE, PSIMITags.ATT_COMPLEX_SUBUNIT_NUM, subUnitNum);
        }
    }

    private String matchIntID (String BINDIntID) {

        if (interactionsMap.containsKey(BINDIntID)) {
            if (badInteractionsMap.containsKey(BINDIntID)) {
                System.err.println ("Error 6: Interaction referenced from complex is a 'bad' interaction.");
                this.badInteractionData = true;
            }
            return interactionsMap.get (BINDIntID);
        } else {
            System.err.println ("Error 7: Bad reference from complex. " +
                   "Interaction does not exist for this species: " + BINDIntID);
            this.badInteractionData = true;
            return BINDIntID;
        }
    }
}
