package org.pathwaycommons.trans.util;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.TreeMap;
import org.jdom.Element;
import org.jdom.JDOMException;

import org.genemania.dw.util.DefParams;
import org.genemania.dw.util.GenUtil;

/**
 * XML stats file entry. Bean like. An instance of this class provides basic
 * info on an xml element, whether in a single file or in a whole set of files:
 * - It will list how many times it is present in each file, if any.
 * - List if it has any children, content (text), or atts. These booleans are
 * set to true if they exist at least once in any of the files.
 * - 'Total Cnt' is same as stats if running a single file.
 * - An element is listed even if there are no uses for it.
 * - The method that reads element data from file does not read stats, and
 * assumes uniqueness in file entries. It only reads the first column (element
 * name) and the last column (full path).
 *
 * @author rashadbadrawi
 */

public class XMLElementInfo implements Comparable {

    private static final String NAME_HEADER = "Element Name";
    private static final String HAS_CHILDREN_HEADER = "Has Children";
    private static final String HAS_ATTS_HEADER = "Has Attributes";
    private static final String HAS_TEXT_HEADER = "Has Text";
    private static final String TOTAL_FILES_HEADER = "Total Files";
    private static final String TOTAL_CNT_HEADER = "Total Count";
    private static final String FULL_PATH_HEADER = "Full Path";

    private static TreeMap <String, XMLElementInfo> xeiMap = new TreeMap <String,
                           XMLElementInfo> ();
    private boolean hasChildrenFlag;
    private boolean hasTextFlag;
    private boolean hasAttsFlag;
    private String fullName;
    private String name;
    private TreeMap <String, Integer> usageMap = new TreeMap <String, Integer> ();

    private XMLElementInfo () {}

    protected XMLElementInfo (Element element, String refName) {

        setFullName(element);
        if (xeiMap.get (getFullName ()) != null) {
            XMLElementInfo xeiOld = xeiMap.get (getFullName());  //same instance
            xeiOld.setHasChildren(element);         //reset some atts, if needed
            xeiOld.setHasAtts(element);
            xeiOld.setHasText(element);
            xeiOld.addUsage(refName, xeiOld.getUsage (refName) + 1);
            xeiMap.put (xeiOld.getFullName(), xeiOld);  //replace existing
        } else {                                        //set the other atts
            setName(element);
            setHasChildren(element);
            setHasAtts (element);
            setHasText (element);
            addUsage (refName, 1);
            xeiMap.put(getFullName (), this);
        }
        //System.out.println (getName() + " " + xeiMap.size());
    }

    public static TreeMap <String, XMLElementInfo> getAll () {

        return xeiMap;
    }

    public static ArrayList <XMLElementInfo> getByName (String elementName) throws
                                             JDOMException {

        GenUtil.validateString(elementName);
        ArrayList <XMLElementInfo> matchingList = new ArrayList <XMLElementInfo> ();
        Iterator iterator = xeiMap.keySet().iterator();
        while (iterator.hasNext()) {
            String key = (String)iterator.next();
            if (elementName.equals (xeiMap.get (key).getName())) {
                matchingList.add (xeiMap.get (key));
            }
        }
        if (matchingList.size () == 0) {
            throw new JDOMException ("Element not found: " + elementName);
        }

        return matchingList;
    }

    protected void setHasChildren (Element element) {

        if (!this.hasChildrenFlag) {
            this.hasChildrenFlag = element.getChildren().size() > 0;
        }
    }

    protected void setHasText (Element element) {

        if (!this.hasTextFlag) {
            this.hasTextFlag = element.getTextTrim().length() > 0;
        }
    }

    protected void setHasAtts (Element element) {

        if (!this.hasAttsFlag) {
            this.hasAttsFlag = element.getAttributes().size() > 0;
        }
    }

    private void setFullName (Element element) {

        if (this.fullName == null) {
            GenUtil.validateNotNull(element);
            fullName = XMLUtil.getPathToRoot(element, "");
        }
    }

    private void setName (Element element) {

        if (this.name == null) {
            GenUtil.validateNotNull(element);
            this.name = element.getQualifiedName();
        }
    }

    protected void addUsage (String refName, int cnt) {

        GenUtil.validateString (refName);
        GenUtil.validatePositiveInt(cnt);
        //if (hasAtts () || hasChildren () || hasText ()) {
            this.usageMap.put (refName, new Integer (cnt));
        //}
    }

    public boolean hasChildren () { return hasChildrenFlag; }

    public boolean hasText () { return hasTextFlag; }

    public boolean hasAtts () { return hasAttsFlag; }

    public String getFullName () { return fullName; }

    public String getName () { return name; }

    public int getUsage (String refName) {

        GenUtil.validateString(refName);
        Integer cntInt = this.usageMap.get (refName);
        if (cntInt != null) {
            return cntInt.intValue();
        }
        //System.out.println ("Element: " + getName() + " does not exist in: " +
        //                    refName);
        return 0;
    }

    public int compareTo (Object obj2) {

        //return this.getName ().compareTo (((XMLStats)obj2).getName ());
        return this.getFullName ().compareTo (((XMLElementInfo)obj2).getFullName ());
    }

    @Override
    public boolean equals (Object obj2) {

        if (obj2 instanceof XMLElementInfo) {
            return this.getFullName ().equals (((XMLElementInfo)obj2).getFullName ());
        } else {
            throw new IllegalArgumentException ("Invalid object type: " +
                                                 obj2.getClass());
        }
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 79 * hash + (this.hasChildrenFlag ? 1 : 0);
        hash = 79 * hash + (this.hasAttsFlag ? 1 : 0);
        hash = 79 * hash + (this.fullName != null ? this.fullName.hashCode() : 0);
        hash = 79 * hash + (this.name != null ? this.name.hashCode() : 0);
        hash = 79 * hash + (this.usageMap != null ? this.usageMap.hashCode() : 0);
        return hash;
    }

    public static String getHeaders () {

       String tempStr = XMLElementInfo.NAME_HEADER;
       tempStr += GenUtil.TAB;

       tempStr += XMLElementInfo.HAS_CHILDREN_HEADER;
       tempStr += GenUtil.TAB;

       tempStr += XMLElementInfo.HAS_ATTS_HEADER;
       tempStr += GenUtil.TAB;

       tempStr += XMLElementInfo.HAS_TEXT_HEADER;
       tempStr += GenUtil.TAB;

       //tempStr += XMLElementInfo.STATS_HEADER;
       //tempStr += GenUtil.TAB;

       tempStr += XMLElementInfo.TOTAL_FILES_HEADER;
       tempStr += GenUtil.TAB;

       tempStr += XMLElementInfo.TOTAL_CNT_HEADER;
       tempStr += GenUtil.TAB;

       tempStr += XMLElementInfo.FULL_PATH_HEADER;
       tempStr += GenUtil.TAB;

       return tempStr;
    }

    public static void readFromFile () throws IOException {

        readFromFile (DefParams.getDefaultProp(DefParams.XMLSTATS_FILE_PROP));
    }

    public static void readFromFile (String statsFileName) throws IOException {

        GenUtil.validateString(statsFileName);
        BufferedReader bw = new BufferedReader (new FileReader (statsFileName));
        String line;
        boolean skipHeader = true;
        while ((line = bw.readLine ()) != null) {
            String tempArr [] = line.split (GenUtil.TAB);
            if (skipHeader) {
                skipHeader = false;
                continue;
            }
            //bypassing setters
            XMLElementInfo xei = new XMLElementInfo ();
            xei.name = tempArr [0];
            xei.hasChildrenFlag = Boolean.parseBoolean(tempArr [1]);
            xei.hasAttsFlag = Boolean.parseBoolean (tempArr [2]);
            xei.hasTextFlag = Boolean.parseBoolean(tempArr [3]);
            xei.fullName = tempArr [tempArr.length - 1];
            xeiMap.put(xei.getFullName (), xei);
        }
        System.out.println ("Done reading xml element info from: " + statsFileName);
    }

    @Override
    public String toString () {

        String tempStr = getName();
        tempStr += GenUtil.TAB;

        tempStr += hasChildren ();
        tempStr += GenUtil.TAB;

        tempStr += hasAtts ();
        tempStr += GenUtil.TAB;

        tempStr += hasText ();
        tempStr += GenUtil.TAB;

        int totalCnt = 0;
        Iterator iterator = usageMap.keySet().iterator();
        while (iterator.hasNext ()) {
            String refName = (String)iterator.next ();
            //tempStr += refName + GenUtil.COLON + usageMap.get (refName).intValue();
            totalCnt += usageMap.get (refName).intValue();
            //tempStr += GenUtil.SEMICOLON;
        }
        //tempStr += GenUtil.TAB;

        tempStr += usageMap.size();
        tempStr += GenUtil.TAB;

        tempStr += totalCnt;
        tempStr += GenUtil.TAB;

        tempStr += getFullName();

        return tempStr;
    }
}
