package org.pathwaycommons.trans.util;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import org.genemania.dw.util.GenUtil;

/**
 * Hosts xml and translation utilities.
 * Utility pattern design.
 * Can also consider xpath for some of the element retrieving utilities.
 * 
 * @author rashadbadrawi
 */

public final class XMLUtil {

    //misc settings
    public static final String XSI_NAME = "xsi";
    public static final String XSI = "http://www.w3.org/2001/XMLSchema-instance";
    public static final String SCHEMA_LOC_ATT = "schemaLocation";
    public static final String XML_FILE_SUFFIX = ".xml";
    public static final String DEF_PARSER = XMLUtil.XERCES_PARSER;   
    //Xerces parser
    public static final String XERCES_PARSER = "org.apache.xerces.parsers.SAXParser";
    //Xerces features
    private static final String PREFIX_XERCES = "http://apache.org/xml/";
    /*
    public static final String SCHEMA_FEATURE_XERCES =
            XMLUtil.PREFIX_XERCES + "features/validation/schema";
    public static final String FULL_CHECK_FEATURE_XERCES =
            XMLUtil.PREFIX_XERCES + "features/validation/schema-full-checking";
    */
    public static final String CONT_FATAL_FEATURE_XERCES =
            XMLUtil.PREFIX_XERCES + "features/continue-after-fatal-error";
    public static final String DYNAMIC_CHECK_FEATURE_XERCES =
            XMLUtil.PREFIX_XERCES + "features/validation/dynamic";
    //Xerces props
    /*
    public static final String SCHEMA_LOC_PROP_XERCES =
            XMLUtil.PREFIX_XERCES + "properties/schema/external-schemaLocation";
    public static final String NONS_SCHEMA_LOC_PROP_XERCES =
            XMLUtil.PREFIX_XERCES + "properties/schema/external-noNamespaceSchemaLocation";
    */
    public static final String BUF_PROP_XERCES =
            XMLUtil.PREFIX_XERCES + "properties/input-buffer-size";
    private static PrintWriter log = GenUtil.getDefaultLog();
    private static SAXBuilder builder;

    private XMLUtil() {}

    public static Element readSchemaBasedXML (String inputFileName) throws
                          JDOMException, IOException {
      
        builder = new SAXBuilder(XMLUtil.DEF_PARSER, false);
        builder.setFeature(XMLUtil.DYNAMIC_CHECK_FEATURE_XERCES, true);
        builder.setFeature(XMLUtil.CONT_FATAL_FEATURE_XERCES, true);   //if necessary
        builder.setProperty(XMLUtil.BUF_PROP_XERCES, new Integer(16));      //better for larger files
        Document doc = builder.build(inputFileName);
        //System.out.println("Processing XML document with root: " + doc.getRootElement().getName());

        return doc.getRootElement();
    }

    public static String getReversePathToRoot(Element element, String name) {

        if (name.length () == 0) {                            //first time
            name = element.getName();
        } else {
            name += GenUtil.DOT + element.getName ();
        }
        if (!(element.getParent () instanceof Document)) {    //hit root
            return getReversePathToRoot ((Element)element.getParent(), name);
        }

        return name;
    }

    public static ArrayList <String> getReversePathAsList (String fullPath) {

        String [] tempArr = fullPath.split(GenUtil.DOUBLE_BACK_SLASH + GenUtil.DOT);
        String [] elementNamesArr = new String [tempArr.length];
        int j = 0;
        for (int i = tempArr.length - 1; i >= 0; i--) {
            elementNamesArr [j++] = tempArr [i];
        }

        return new ArrayList <String> (Arrays.asList((elementNamesArr)));
    }

    public static String getPathToRoot (Element element, String name) {

        String fullPath = XMLUtil.getReversePathToRoot(element, name);
        ArrayList <String> tempList = getReversePathAsList (fullPath);
        fullPath = "";
        for (int i = 0; i < tempList.size (); i++) {
           fullPath += tempList.get (i);
           if (i != tempList.size () - 1) {
               fullPath += GenUtil.DOT;
           }
        }

        return fullPath;
    }

    public static void dump (Element rootElement, String fileName)
                            throws IOException {

       BufferedWriter bw = new BufferedWriter (new FileWriter (fileName));
       dump (rootElement, bw);
       System.out.println ("Dumped xml file: " + fileName);
       bw.flush();
       bw.close();
    }

    public static void dump (Element rootElement, BufferedWriter bw) throws
                                                               IOException {

       Format formatter = Format.getPrettyFormat();
       formatter.setExpandEmptyElements(true);
       formatter.setOmitDeclaration(false);
       XMLOutputter xmlPrint = new XMLOutputter (formatter);
       xmlPrint.output(rootElement, bw);
    }

    public static boolean validateExists (String elementName, ArrayList <Element>
                                          elementList) {

        if (elementList.size () == 0) {
            //System.err.println ("Wrong number of elements " +
            //        elementName + " " + elementList.size () + " (expected: 1-*)");
        }

        return elementList.size () > 0;
    }

    public static boolean validateSingle (String elementName, ArrayList <Element>
                                          elementList) {

        if (elementList.size () == 0 || elementList.size () > 1) {
            //System.err.println ("Wrong number of elements " +
            //        elementName + " " + elementList.size () + " (expected: 1)");
            for (int i = 0; i < elementList.size (); i++) {
                //System.err.println (elementList.get (i).getTextTrim() + " " +
                //XMLUtil.getPathToRoot(elementList.get (i), ""));
            }
        }

        return elementList.size () > 0;
    }

    public static boolean validateInt (Element e) {

        if (e.getTextTrim().length() > 0) {
            if (Integer.parseInt(e.getTextTrim()) > 0) {
                return true;
            }
        }
        //System.err.println ("Invalid integer value for element: " +
        //                    e.getName() + " " + e.getTextTrim () + " " +
        //                    e.getTextTrim().length());
        
        return false;
    }

    //supports unique elements only
    public static int getDepth (String elementName) throws JDOMException {

        ArrayList <XMLElementInfo> xeiList = XMLElementInfo.getByName (elementName);
        if (xeiList.size () > 1) {
            System.err.println ("Warning 1 - element List: Multiple occurences: "
                                + elementName);
        }
        String [] parentsArr = xeiList.get (0).getFullName().split(
                                      GenUtil.DOUBLE_BACK_SLASH + GenUtil.DOT);

        return parentsArr.length;
    }

    public static ArrayList <Element> getElement (String elementName, Element
            parent, int index) throws JDOMException {

        return getElement (elementName, parent, index, false);
    }

    //multipleFlag allows several paths to the same parent-distantChild pair,
    //like PubMed IDs in BIND.
    //parent represents the nearest parent that allows identifying a unique
    //path to the specified element. 
    public static ArrayList <Element> getElement (String elementName, Element
               parent, int index, boolean multipleFlag) throws JDOMException {

        //quality control section
        ArrayList <XMLElementInfo> xeiList = XMLElementInfo.getByName (elementName);
        XMLElementInfo xei = null;
        int cnt = 0;
        for (int i = 0; i < xeiList.size (); i++) {
            if (xeiList.get (i).getFullName().contains(parent.getName())) {
                //System.out.println (xeiList.get (i).getName() + "..." + i);
                xei = xeiList.get (i);
                cnt++;
            }
        }
        if (cnt > 1 && !multipleFlag) {
            System.err.println ("Warning 2 - element List: Multiple occurences: " +
                    elementName + " under: " + parent.getName() + " " + xeiList.size());
        }
        if (cnt == 0) {
            System.err.println ("Warning 3 - element List: No such element: " +
                    elementName + " under " + parent.getName());
        }
        //end section
        ArrayList <Element> tempList = new ArrayList <Element> ();
        if (multipleFlag) {
            for (int i = 0; i < xeiList.size (); i++) {
                xei = xeiList.get (i);
                //System.out.println ("multiple elements: " +
                //        parent.getName() + " " + xei.getFullName());
                String [] parentsArr = xei.getFullName().split(
                                       GenUtil.DOUBLE_BACK_SLASH + GenUtil.DOT);
                tempList.addAll(getTargetChildren (parent, parentsArr, index));
            }
        } else {
            String [] parentsArr = xei.getFullName().split(
                       GenUtil.DOUBLE_BACK_SLASH + GenUtil.DOT);
            tempList.addAll(getTargetChildren (parent, parentsArr, index));
        }
        if (tempList.size() == 0) {
            //log.println ("Not found: " + parent.getName () + " " + elementName);
        } else if (!tempList.get (0).getName().equals(elementName)) {
            System.err.println ("Error: Wrong Element retrieved " +
                               tempList.get (0) + " " + elementName);
        }

        return tempList;
    }

    //no support for multiple children.
    private static ArrayList <Element> getTargetChildren (Element parent,
                                              String [] parentsArr,int index) {

        ArrayList <Element> tempList = new ArrayList <Element> ();
        tempList.addAll (parent.getChildren (parentsArr [index]));
        for (int i = 0; i < tempList.size (); i++) {
            //System.out.println (tempList.get (i) + " " + index + " " +
            //                    parentsArr.length  + " " + parentsArr [index]);
            if (index < parentsArr.length - 1) {
               return getTargetChildren (tempList.get (i), parentsArr, ++index);
            }
        }

        return tempList;
    }
}
