function [alpha,bias,Yneg,kesi,kesi_bar]=SemiSVR_PolyKernel_LP(KM1,KM2,Y,y,C1,C2,eps,method)
%
% Semi-quantitative eSVR for Linear & nonlinear Regression with User-defined polynomial Kernel.
%
% Copyright 2009, Xiaojian SHAO
% USAGE:
% [alpha,bias,Yneg,kesi,kesi_bar]=SemiSVR_PolyKernel_LP(KM1,KM2,Y,y,C1,C2,eps,method)
%
%  %INPUT%
%%%%%%%%%%%%%%%%%%%%
% KM1    : The kernel matrix on the positive samples, m*m matrix;
% KM2    : The kernel matrix on the negative samples, k*m matrix;
%          the column for KM1 and KM2 must be the same;
% Y      : The regression value on the positive data;
% y      : The threshold for the negative qualitative data;
%         that's mean regression(negative pairs)>=y;
% C1, C2 : The penalty parameters for positive and negative samples respectively;
% eps    : The parameter of the epsilon-insensitive loss function;
% method : The method for solving the LP, a string of the form 'Simplex' and
%           'Interior'; default 'Interior'.
%
% % OUTPUT %
%%%%%%%%%%%%%%%%%%%%
% alpha          : The Lagrange multiplier;
% bias              : The bias of the regression function;
% Yneg           : The inferred regression value on negative qualitative data;
% Kesi, Kesi_bar : The corresponding slack varible values for postive and negative data.
%
%
%%%%%%%% Decision function %
% f(x)=PolyKernel(x,Xtrain)*alpha - b;
%
% latent variables: alpha = u-v;
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Set the default method for solving the LP.
if nargin < 8 
    method = 'Interior';
end
if nargin < 7
    help SemiSVR_PolyKernel_LP
    error('Not enough input for the program!')
end

m1=size(KM1,1);e1=ones(m1,1);
m2=size(KM2,1);e2=ones(m2,1);
m=m1+m2;

%%%%%%%
%%% alpha=u-v; u>=0,v>=0;
f=[ [e1;e1]; zeros(m2+1,1); C1*e1; C2*e2];
%% Linear object function : e'(u+v) + 0*Yneg + 0*b + C1*e1'kesi + C2*e2'kesi_bar

A1=[-KM1 KM1 e1 zeros(m1,m2)  -eye(m1) zeros(m1,m2)];
%%% -KM1*alpha + e1'*b - I*kesi + 0*kesi_bar <= -Y+eps;

A2=[KM1 -KM1 -e1 zeros(m1,m2) -eye(m1) zeros(m1,m2)];
%% KM1*alpha - e1*b - I*kesi + 0*kesi_bar <= Y+eps;

A3=[-KM2 KM2 e2 eye(m2)  zeros(m2,m1) -eye(m2)];
%% -KM2*alpha + b*e2 + I*Yneg + 0*kesi - I*kesi_bar <= +(eps*e2);

A4=[KM2 -KM2 -e2 -eye(m2)  zeros(m2,m1) -eye(m2)];
%% KM2*alpha - b*e2 - I*Yneg + 0*kesi - I*kesi_bar <= +(eps*e2);

%% Construct Matrix for linear inequality constraints;
A=[A1;A2;A3;A4];
A=sparse(A);

%% Construct Vector for linear inequality constraints
b =[ -Y+eps*e1; Y+eps*e1; eps*e2; eps*e2];

%% Vector of lower bounds
lb=[zeros(2*m1,1); -inf; y*e2; zeros(m,1)];
%% u,v >= 0; bias: free, Yneg>=y*e2,kesi>=0; kesi_bar>=0;

clear A1 A2 A3 A4  KM1 KM2 e1 e2 
%% Clear not used variable to free memory for solving LP.

switch method
    case 'Simplex'
        ub=inf;
        options = optimset('LargeScale', 'off', 'Simplex', 'on'); 
        
        %% Using Simplex method to solve LP;
        [x, fval, exitflag] = linprog(f,A,b,[],[],lb,ub,[],options);
    case 'Interior'
        %% Using interior points method to solve LP;
        [x, fval, exitflag] = linprog(f,A,b,[],[],lb); 
    otherwise
        error('Please choose the method to slove the LP: Simplex or Interior!')
end

s=sprintf('The fval is %f; The terminated flag is %d. \n',fval,exitflag);
disp(s);

%%% Got the final results of different varialbes from the solution of LP.
alpha=x(1:m1)-x(m1+1:2*m1);
bias=x(2*m1+1);
Yneg=x(2*m1+2:2*m1+m2+1);
kesi=x(2*m1+m2+2:3*m1+m2+1);
kesi_bar=x(3*m1+m2+2:3*m1+2*m2+1);
