package org.baderlab.pdzsvm.analysis;

import org.baderlab.pdzsvm.data.DataLoader;
import org.baderlab.pdzsvm.data.manager.DataFileManager;
import org.baderlab.pdzsvm.utils.PDZSVMUtils;

import java.util.List;
import java.util.ArrayList;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.File;

import org.baderlab.pdzsvm.utils.Constants;
import org.baderlab.pdzsvm.predictor.Predictor;
import org.baderlab.pdzsvm.predictor.svm.ContactMapSVMPredictor;
import org.baderlab.pdzsvm.predictor.svm.GlobalSVMPredictor;
import org.baderlab.pdzsvm.evaluation.Evaluation;
import weka.core.Utils;
import weka.core.Instances;
import libsvm.svm_parameter;

/**
 * Copyright (c) 2010 University of Toronto
 * Code written by: Shirley Hui
 * Authors: Shirley Hui, Gary Bader
 *
 * This file is part of PDZSVM.
 *
 * PDZSVM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PDZSVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * documentation provided hereunder is on an "as is" basis, and the
 * University of Toronto has no obligations to provide maintenance,
 * support, updates, enhancements or modifications.  In no event shall
 * the University of Toronto be liable to any party for direct, indirect,
 * special, incidental or consequential damages, including lost profits,
 * arising out of the use of this software and its documentation, even if
 * the University of Toronto has been advised of the possibility of such
 * damage. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PDZSVM.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * This file compares SVMs trained using different feature encoding methods
 * discussed in the paper.  Produces R files for Fig. 3b.
 */
public class CompareFeatureEncodings {
    private String testName = "";
    private String dirName ;
    private boolean print = true;
    private List posTestProfileList;
    private List negTestProfileList;
    private DataLoader dl;
    private String parentDir = "/FeatureEncoding";

    public void loadTests(int test)
    {
        dl.clearTest();
        if (test==5)
        {
            dl.loadMouseTest(Constants.MOUSE_ORPHAN);  testName = "MOUSE ORPHAN PM";
            dirName = "/PMMouseOrphanG";

        }
        else if (test==6)
        {
            dl.loadWormTest(Constants.PROTEIN_MICROARRAY); testName ="WORM PM";
            dirName = "/PMWormG";

        }
        else if (test==7)
        {
            dl.loadFlyTest(); testName = "FLY PM";
            dirName = "/PMFlyG";

        }

        posTestProfileList = dl.getPosTestProfileList();
        negTestProfileList= dl.getNegTestProfileList();

        System.out.println("\tTEST IS: " + testName);


    }
    public CompareFeatureEncodings()
    {
        dl = new DataLoader();
    }
    public static  void main(String[] args)
    {
        CompareFeatureEncodings cp = new CompareFeatureEncodings();
        int testNum = Integer.parseInt(args[0]);
        cp.compare(testNum);

    }

    private void print(String output, String fileName)
    {
        System.out.println("\tWriting to " + DataFileManager.OUTPUT_ROOT_DIR + parentDir + dirName + "/" + fileName + "...");
        
        try
        {
            BufferedWriter bw = new BufferedWriter(new FileWriter(new File(DataFileManager.OUTPUT_ROOT_DIR + parentDir + dirName + "/" + fileName)));
            bw.write(output);
            bw.close();
        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
        }
    }
    private void compare(int testNum)
    {
        List globalSVMBSQPredictions = new ArrayList();
        List globalSVMPCPredictions = new ArrayList();
        List globalSVMCMPredictions = new ArrayList();
        loadTests(testNum);

        globalSVMBSQPredictions.addAll(runSVMBSQ());
        globalSVMPCPredictions.addAll(runSVMPC());
        globalSVMCMPredictions.addAll(runSVMCM());

        List predList =new ArrayList();
        List rocAUCList = new ArrayList();
        List prAUCList = new ArrayList();

        List aucLabelList = new ArrayList();
        String predictorName = "SVM BSQ";
        aucLabelList.add(predictorName);
        Evaluation evalBSQ = evaluate(globalSVMBSQPredictions, predictorName);
        double rocAUC = Double.parseDouble(Utils.doubleToString(evalBSQ.getROCAUC(),3));
        rocAUCList.add(rocAUC);
        double prAUC = Double.parseDouble(Utils.doubleToString(evalBSQ.getPRAUC(),3));
        prAUCList.add(prAUC);
        Instances instBSQ = evalBSQ.getCurve(1);
        predList.add(instBSQ);
        if (print)
        {
            List predictions = new ArrayList();
            predictions.add(globalSVMBSQPredictions);

            StringBuffer rString = PDZSVMUtils.toRString(predictions);
            print(rString.toString(),"test_" + predictorName.replace(' ','_') + "_Load.r");

        }

        predictorName = "SVM PC";
        aucLabelList.add(predictorName);
        Evaluation evalPC = evaluate(globalSVMPCPredictions, predictorName);
        rocAUC = Double.parseDouble(Utils.doubleToString(evalPC.getROCAUC(),3));
        rocAUCList.add(rocAUC);
        prAUC = Double.parseDouble(Utils.doubleToString(evalPC.getPRAUC(),3));
        prAUCList.add(prAUC);
        Instances instPC = evalPC.getCurve(1);
        predList.add(instPC);
        if (print)
        {
            List predictions = new ArrayList();
            predictions.add(globalSVMPCPredictions);

            StringBuffer rString = PDZSVMUtils.toRString(predictions);
            print(rString.toString(),"test_" + predictorName.replace(' ','_') + "_Load.r");

        }

        predictorName = "SVM CM";
        aucLabelList.add(predictorName);
        Evaluation evalCM = evaluate(globalSVMCMPredictions, predictorName);
        rocAUC = Double.parseDouble(Utils.doubleToString(evalCM.getROCAUC(),3));
        rocAUCList.add(rocAUC);
        prAUC = Double.parseDouble(Utils.doubleToString(evalCM.getPRAUC(),3));
        prAUCList.add(prAUC);
        Instances instCM = evalCM.getCurve(1);
        predList.add(instCM);
        if (print)
        {
            List predictions = new ArrayList();
            predictions.add(globalSVMCMPredictions);

            StringBuffer rString = PDZSVMUtils.toRString(predictions);
            print(rString.toString(),"test_" + predictorName.replace(' ','_') + "_Load.r");
        }

        Predictor.plotCurves(predList, rocAUCList, prAUCList, aucLabelList, testName);

    }

    private Evaluation evaluate(List predictionList, String predictorName)
    {
        Evaluation eval = new Evaluation(predictionList);

        System.out.println("=== Summary " +testName+ " ("+predictorName+") ===");
        System.out.println(eval.toString());
        return eval;
    }

    private List runSVMPC()
    {
        DataLoader dl = new DataLoader();
        dl.loadMouseChenTrain();
        dl.loadHumanTrain(Constants.PWM);

        List posTrainProfileList = dl.getPosTrainProfileList();
        List negTrainProfileList = dl.getNegTrainProfileList();

        svm_parameter svmParams = new svm_parameter();
        svmParams.setDefaults();
        double C = 5; double g = 2;

        System.out.println("\tSVM PC, [g,C] = ["+g+","+C+"])");
        svmParams.C = Math.exp(C);
        svmParams.gamma = Math.exp(-Math.log(2)-g);
        svmParams.data_encoding = svm_parameter.PHYSICOCHEMICAL;

        GlobalSVMPredictor gp =new GlobalSVMPredictor(posTrainProfileList,
                negTrainProfileList,
                svmParams);
        gp.train();
        List predictionList = gp.predict(posTestProfileList,negTestProfileList);

        return predictionList;
    }
    private List runSVMBSQ()
    {
        DataLoader dl = new DataLoader();
        dl.loadMouseChenTrain();
        dl.loadHumanTrain(Constants.PWM);

        List posTrainProfileList = dl.getPosTrainProfileList();
        List negTrainProfileList = dl.getNegTrainProfileList();

        svm_parameter svmParams = new svm_parameter();
        svmParams.setDefaults();

        svmParams.data_encoding = svm_parameter.BINARY_SEQUENCE;
        svmParams.C = Math.exp(2);
        svmParams.gamma = Math.exp(-Math.log(2)-2);

        GlobalSVMPredictor gp =new GlobalSVMPredictor(posTrainProfileList,
                negTrainProfileList,
                svmParams);
        gp.train();
        List predictions = gp.predict(posTestProfileList,negTestProfileList);

        return predictions;
    }
    private List runSVMCM()
    {
        DataLoader dl = new DataLoader();
        dl.loadMouseChenTrain();
        dl.loadHumanTrain(Constants.PWM);

        List posTrainProfileList = dl.getPosTrainProfileList();
        List negTrainProfileList = dl.getNegTrainProfileList();

        svm_parameter svmParams = new svm_parameter();
        svmParams.setDefaults();
        double C = 2; double g = 4;

        System.out.println("\tSVM CM, [g,C] = ["+g+","+C+"])");
        svmParams.C = Math.exp(C);
        svmParams.gamma = Math.exp(-Math.log(2)-g);
        svmParams.data_encoding = svm_parameter.CONTACTMAP2020;

        ContactMapSVMPredictor c =new ContactMapSVMPredictor(
                posTrainProfileList,
                negTrainProfileList,
                svmParams);
        c.train();
        List predictions = c.predict(posTestProfileList, negTestProfileList);

        return predictions;

    }

}
