package org.baderlab.pdzsvm.encoding;

import java.util.*;

import org.biojava.bio.seq.Sequence;
import org.baderlab.pdzsvm.utils.Constants;
import org.baderlab.pdzsvm.utils.PDZSVMUtils;
import org.baderlab.pdzsvm.data.manager.DataFileManager;

/**
 * Copyright (c) 2010 University of Toronto
 * Code written by: Shirley Hui
 * Authors: Shirley Hui, Gary Bader
 *
 * This file is part of PDZSVM.
 *
 * PDZSVM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PDZSVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * documentation provided hereunder is on an "as is" basis, and the
 * University of Toronto has no obligations to provide maintenance,
 * support, updates, enhancements or modifications.  In no event shall
 * the University of Toronto be liable to any party for direct, indirect,
 * special, incidental or consequential damages, including lost profits,
 * arising out of the use of this software and its documentation, even if
 * the University of Toronto has been advised of the possibility of such
 * damage. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PDZSVM.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Chen 16 encoding represents only 16 domain positions lying in the binding
 * site which were found to be in contact (< 5.0 angstroms) to positions in
 * the peptide based on the three dimensional structure of mouse alpha1-syntrophin
 * in complex with a heptapeptide.
 * Chen, J. et al. (2008) Predicting PDZ domain-peptide interactions from primary
 *   sequences, Nat. Biotechnol., 26, 1041-1045.
 */
public class Chen16FeatureEncoding extends FeatureEncoding
{
    public final static int numContacts = 38;

    public int[][] contactPositions = new int[numContacts][2];
    public HashMap flyDomainSeqToSequence16Map = new HashMap();
    public HashMap mouseDomainSeqToSequence16Map = new HashMap();
    public HashMap humanDomainSeqToSequence16Map = new HashMap();
    public HashMap wormDomainSeqToSequence16Map = new HashMap();

    public Chen16FeatureEncoding()
    {
        setEncodingName("CHEN16");

        String humanFile = DataFileManager.DATA_ROOT_DIR + "/Data/Chen/HumanPDZDomainSeq16Positions.fa";
        String humanBlindFile = DataFileManager.DATA_ROOT_DIR + "/Data/Chen/HumanPDZBlindDomainSeq16Positions.fa";
        String humanPDZScanFile = DataFileManager.DATA_ROOT_DIR + "/Data/Chen/HumanPDZScanDomainSeq16Positions.fa";

        loadHumanMap(humanFile);
        loadHumanMap(humanBlindFile);
        loadHumanMap(humanPDZScanFile);


        String mouseFile =  DataFileManager.DATA_ROOT_DIR  + "/Data/Chen/MousePDZDomainSeq16Positions.fa";
        String mouseOrphanFile =  DataFileManager.DATA_ROOT_DIR  + "/Data/Chen/MouseOrphanPDZDomainSeq16Positions.fa";

        loadMouseMap(mouseFile);
        loadMouseMap(mouseOrphanFile);

        String wormFile =  DataFileManager.DATA_ROOT_DIR  + "/Data/Chen/WormPDZDomainSeq16Positions.fa";
        String wormChenFile =  DataFileManager.DATA_ROOT_DIR  + "/Data/Chen/WormChenPDZDomainSeq16Positions.fa";
        loadWormMap(wormFile);
        loadWormMap(wormChenFile);

        String flyFile =  DataFileManager.DATA_ROOT_DIR  + "/Data/Chen/FlyChenPDZDomainSeq16Positions.fa";
        loadFlyMap(flyFile);

        loadContactPositions();
    }

    protected List encodeFeature(String value)
    {
        // Does nothing
        return null;
    }
    public String getFeatures(String domainSeq, String organism)
    {
        Sequence domain16Sequence;
        String domain16SeqString;
        if (organism.equals(Constants.MOUSE))
            domain16Sequence =  (Sequence)mouseDomainSeqToSequence16Map.get(domainSeq);
        else if (organism.equals(Constants.HUMAN))
            domain16Sequence =  (Sequence)humanDomainSeqToSequence16Map.get(domainSeq);
        else if (organism.equals(Constants.WORM))
            domain16Sequence =  (Sequence)wormDomainSeqToSequence16Map.get(domainSeq);
        else if (organism.equals(Constants.FLY))
            domain16Sequence =  (Sequence)flyDomainSeqToSequence16Map.get(domainSeq);
        else
            return null;
        if (domain16Sequence == null)
        {
            System.out.println("\tCould not encode: " + domainSeq);
            return null;
        }
        domain16SeqString= domain16Sequence.seqString();
        return domain16SeqString;
    }
    private void loadFlyMap(String flyFile)
    {
        try
        {
            HashMap tempMap = PDZSVMUtils.readAlignment(flyFile);
            Set keys = tempMap.keySet();
            Iterator it = keys.iterator();
            while(it.hasNext())
            {
                String seqName = (String)it.next();
                Sequence seq =  (Sequence)tempMap.get(seqName);
                flyDomainSeqToSequence16Map.put(seqName, seq);
            }
        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
          
        }

    }

    private void loadMouseMap(String mouseFile)
    {
        try
        {
            HashMap tempMap = PDZSVMUtils.readAlignment(mouseFile);
            Set keys = tempMap.keySet();
            Iterator it = keys.iterator();
            while(it.hasNext())
            {
                String seqName = (String)it.next();
                Sequence seq =  (Sequence)tempMap.get(seqName);
                mouseDomainSeqToSequence16Map.put(seqName, seq);
            }
        }
        catch(Exception e)
        {

        }
    }
    private void loadWormMap(String wormFile)
    {
        try
        {
            HashMap tempMap = PDZSVMUtils.readAlignment(wormFile);
            Set keys = tempMap.keySet();
            Iterator it = keys.iterator();
            while(it.hasNext())
            {
                String seqName = (String)it.next();
                Sequence seq =  (Sequence)tempMap.get(seqName);
                wormDomainSeqToSequence16Map.put(seqName, seq);
            }
        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);

        }

    }


    private void loadHumanMap(String humanFile)
    {
        try
        {
            HashMap tempMap = PDZSVMUtils.readAlignment(humanFile);
            Set keys = tempMap.keySet();
            Iterator it = keys.iterator();
            while(it.hasNext())
            {
                String seqName = (String)it.next();
                Sequence seq =  (Sequence)tempMap.get(seqName);
                humanDomainSeqToSequence16Map.put(seqName, seq);
            }
        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
        }
    }

    private void loadContactPositions()
       {

           contactPositions[0] = new int[]{0,4};
           contactPositions[1] = new int[]{1,4};
           contactPositions[2] = new int[]{2,3};
           contactPositions[3] = new int[]{2,4};
           contactPositions[4] = new int[]{3,2};
           contactPositions[5] = new int[]{3,3};
           contactPositions[6] = new int[]{3,4};
           contactPositions[7] = new int[]{4,1};
           contactPositions[8] = new int[]{4,2};
           contactPositions[9] = new int[]{4,3};
           contactPositions[10] = new int[]{4,4};
           contactPositions[11] = new int[]{5,0};
           contactPositions[12] = new int[]{5,1};
           contactPositions[13] = new int[]{5,2};
           contactPositions[14] = new int[]{5,3};
           contactPositions[15] = new int[]{5,4};
           contactPositions[16] = new int[]{6,0};
           contactPositions[17] = new int[]{6,1};
           contactPositions[18] = new int[]{6,2};
           contactPositions[19] = new int[]{7,0};
           contactPositions[20] = new int[]{7,1};
           contactPositions[21] = new int[]{8,1};
           contactPositions[22] = new int[]{9,1};
           contactPositions[23] = new int[]{9,3};
           contactPositions[24] = new int[]{10,2};
           contactPositions[25] = new int[]{10,3};
           contactPositions[26] = new int[]{10,4};
           contactPositions[27] = new int[]{11,3};
           contactPositions[28] = new int[]{11,4};
           contactPositions[29] = new int[]{12,0};
           contactPositions[30] = new int[]{12,1};
           contactPositions[31] = new int[]{12,2};
           contactPositions[32] = new int[]{13,0};
           contactPositions[33] = new int[]{14,0};
           contactPositions[34] = new int[]{14,2};
           contactPositions[35] = new int[]{14,3};
           contactPositions[36] = new int[]{14,4};
           contactPositions[37] = new int[]{15,4};
       }


}
