package org.baderlab.pdzsvm.predictor.nn;

import java.util.*;
import java.util.List;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.File;

import org.baderlab.pdzsvm.data.*;
import org.baderlab.brain.ProteinProfile;
import org.biojava.bio.seq.Sequence;
import org.baderlab.pdzsvm.evaluation.Prediction;
import org.baderlab.pdzsvm.evaluation.Evaluation;
import org.baderlab.pdzsvm.encoding.Chen16FeatureEncoding;
import org.baderlab.pdzsvm.encoding.Features;
import org.baderlab.pdzsvm.predictor.svm.ContactMapSVMPredictor;
import org.baderlab.pdzsvm.predictor.Predictor;
import org.baderlab.pdzsvm.validation.ValidationParameters;
import org.baderlab.pdzsvm.validation.ValidationFoldOutput;
import org.baderlab.pdzsvm.utils.PDZSVMUtils;
import org.baderlab.pdzsvm.utils.Constants;
import weka.core.Instances;

/**
 * Copyright (c) 2010 University of Toronto
 * Code written by: Shirley Hui
 * Authors: Shirley Hui, Gary Bader
 *
 * This file is part of PDZSVM.
 *
 * PDZSVM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PDZSVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * documentation provided hereunder is on an "as is" basis, and the
 * University of Toronto has no obligations to provide maintenance,
 * support, updates, enhancements or modifications.  In no event shall
 * the University of Toronto be liable to any party for direct, indirect,
 * special, incidental or consequential damages, including lost profits,
 * arising out of the use of this software and its documentation, even if
 * the University of Toronto has been advised of the possibility of such
 * damage. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PDZSVM.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Nearest neighbour predictor which makes positive or negative predictions
 * based on whether or not the nearest training neighbour for a given
 * domain peptide interaction is positive or negative.  Nearest neighbour is
 * determined by Hamming distance using Chen16FeatureEncoding.
 */
public class NNPredictor extends Predictor
{
    private List posTrainIntSeqList;
    private List negTrainIntSeqList;
    private int encoding = ContactMapSVMPredictor.CHEN16;

    // Uses only contact map encoding
    public NNPredictor(List posTrainProfileList,
                       List negTrainProfileList)
    {
        super(posTrainProfileList, negTrainProfileList);
        System.out.println("\tNN Binding Site...");
        predictorName = "NN BS";


    }

    public void train()
    {
        System.out.println("\tTraining...");
        Chen16FeatureEncoding enc = new Chen16FeatureEncoding();

        posTrainIntSeqList = new ArrayList();
        negTrainIntSeqList = new ArrayList();

        for (int i = 0;i < posTrainProfileList.size();i++)
        {
            ProteinProfile posProfile = (ProteinProfile)posTrainProfileList.get(i);
            String domainSeq = posProfile.getDomainSequence();

            String organismLong = posProfile.getOrganism();
            String organism = PDZSVMUtils.organismLongToShortForm(organismLong);
            domainSeq = enc.getFeatures(domainSeq, organism);

            Collection seqCollection = posProfile.getSequenceMap();
            List seqList  = new ArrayList(seqCollection);
            for (int j = 0; j < seqList.size();j++)
            {
                Sequence seq = (Sequence)seqList.get(j);
                String peptideSeq = seq.seqString();
                String posIntString = domainSeq + peptideSeq;
                posTrainIntSeqList.add(posIntString);
            }

        }
        for (int i = 0;i < negTrainProfileList.size();i++)
        {
            ProteinProfile negProfile = (ProteinProfile)negTrainProfileList.get(i);
            if (negProfile==null)
                continue;
            String domainSeq = negProfile.getDomainSequence();

            String organismLong = negProfile.getOrganism();
            String organism = PDZSVMUtils.organismLongToShortForm(organismLong);
            domainSeq = enc.getFeatures(domainSeq, organism);

            Collection seqCollection = negProfile.getSequenceMap();
            List seqList  = new ArrayList(seqCollection);
            for (int j = 0; j < seqList.size();j++)
            {
                Sequence seq = (Sequence)seqList.get(j);
                String peptideSeq = seq.seqString();
                String negIntString = domainSeq + peptideSeq;
                negTrainIntSeqList.add(negIntString);
            }
        }

    }

    public List predict(List posTestProfileList, List negTestProfileList)
    {
        // Clear prediction list!
        predictionList = new ArrayList();
        Data testData = new Data();
        testData.addRawData(posTestProfileList, Constants.CLASS_YES);
        if (!negTestProfileList.isEmpty() && negTestProfileList !=  null)
            testData.addRawData(negTestProfileList,Constants.CLASS_NO);
        Chen16FeatureEncoding enc = new Chen16FeatureEncoding();

        List balPosTestProfileList = new ArrayList();
        List balNegTestProfileList = new ArrayList();
        HashMap posTestProfileHashMap = PDZSVMUtils.profileListToHashMap(posTestProfileList);
        HashMap negTestProfileHashMap = PDZSVMUtils.profileListToHashMap(negTestProfileList);
        for (int i =0; i < negTestProfileList.size();i++)
        {
            ProteinProfile negProfile = (ProteinProfile)negTestProfileList.get(i);
            ProteinProfile posProfile = (ProteinProfile)posTestProfileHashMap.get(negProfile.getName());
            if (posProfile == null)
            {
                balNegTestProfileList.add(negProfile);
                balPosTestProfileList.add(null);
            }
            else
            {
                balNegTestProfileList.add(negProfile);
                balPosTestProfileList.add(posProfile);
            }
        }
        for (int i =0; i < posTestProfileList.size();i++)
        {
            ProteinProfile posProfile = (ProteinProfile)posTestProfileList.get(i);
            ProteinProfile negProfile = (ProteinProfile)negTestProfileHashMap.get(posProfile.getName());
            if (negProfile == null)
            {
                balNegTestProfileList.add(null);
                balPosTestProfileList.add(posProfile);
            }

        }
        ProteinProfile profile;
        for (int i =0; i < balPosTestProfileList.size();i++)
        {
            ProteinProfile testPosProfile = (ProteinProfile)balPosTestProfileList.get(i);
            ProteinProfile testNegProfile = (ProteinProfile)balNegTestProfileList.get(i);

            if (testPosProfile !=null)
                profile = testPosProfile;
            else
                profile = testNegProfile;
            String organismLong = profile.getOrganism();
            String organism = PDZSVMUtils.organismLongToShortForm(organismLong);
            String name = profile.getName();
            String methodLong = profile.getExperimentalMethod();
            String method = PDZSVMUtils.methodLongToShortForm(methodLong);
            String domainSeqFull = profile.getDomainSequence();

            String domainSeq = enc.getFeatures(domainSeqFull, organism);

            if (testPosProfile!=null)
            {
                Collection seqCollection = testPosProfile.getSequenceMap();
                List seqList  = new ArrayList(seqCollection);
                for (int j = 0; j < seqList.size();j++)
                {
                    NNInfo posNN = null;
                    NNInfo negNN= null;
                    Prediction pred;
                    Sequence seq = (Sequence)seqList.get(j);
                    String peptideSeq = seq.seqString();

                    String intSeqString = domainSeq + peptideSeq;

                    // Find NN neighbour interaction in pos or neg int seq lists
                    posNN = NN.getNNBindingSiteSeqInfo(intSeqString, posTrainIntSeqList);
                    negNN= NN.getNNBindingSiteSeqInfo(intSeqString, negTrainIntSeqList);


                    double decValue = posNN.sim - negNN.sim;
                    // True positive
                    if (decValue >0 )
                    {
                        pred  = new Prediction(1.0,1,decValue, name, domainSeqFull, peptideSeq, organism, method);
                    }
                    else
                    {
                        pred  = new Prediction(0.0,1,decValue, name,domainSeqFull, peptideSeq, organism, method );
                    }
                    predictionList.add(pred);

                }
            }
            if (testNegProfile != null)
            {
                Collection seqCollection = testNegProfile.getSequenceMap();
                List seqList  = new ArrayList(seqCollection);
                for (int j = 0; j < seqList.size();j++)
                {

                    Prediction pred;
                    Sequence seq = (Sequence)seqList.get(j);
                    String peptideSeq = seq.seqString();


                    String intSeqString = domainSeq + peptideSeq;
                    // Find NN neighbour interaction in pos or neg int seq lists
                    NNInfo posNN= NN.getNNBindingSiteSeqInfo(intSeqString, posTrainIntSeqList);
                    NNInfo negNN= NN.getNNBindingSiteSeqInfo(intSeqString, negTrainIntSeqList);

                    double decValue = posNN.sim - negNN.sim;

                    // True negative
                    if (decValue<=0)
                    {
                        pred  = new Prediction(0.0,0,decValue, name,domainSeqFull, peptideSeq, organism, method );
                    }
                    else
                    {
                        pred  = new Prediction(1.0,0,decValue, name,domainSeqFull, peptideSeq, organism, method );
                    }

                    predictionList.add(pred);

                }

            }
        }
        Evaluation eval = new Evaluation(predictionList);
        double rocAUC = eval.getROCAUC();
        if (rocAUC<0.5)
        {
            System.out.println("\tFlipping signs...");

            for (int ii = 0; ii < predictionList.size();ii++)
            {
                Prediction prediction = (Prediction)predictionList.get(ii);
                double decValue = prediction.getDecValue();
                prediction.setDecValue(-decValue);
            }
        }
        return predictionList;
    }
    public List getPredictions()
    {
        return predictionList;
    }
    public static void main(String[] args)
    {
        DataLoader dl = new DataLoader();

        dl.loadMouseChenTrain();
        dl.loadHumanTrain(Constants.PWM);

        List posTrainProfileList = dl.getPosTrainProfileList();
        List negTrainProfileList = dl.getNegTrainProfileList();

        String testName = "";
        String dirName = "";
        dl.loadWormTest(Constants.PROTEIN_MICROARRAY); testName = "WORM PM";  dirName = "PMWormG";
        //dl.loadMouseTest("ORPHAN"); testName = "MOUSE ORPHAN PM";  dirName = "PMMouseOrphanG";
        //dl.loadFlyTest(); testName = "FLY PM";  dirName = "PMFlyG";

        NNPredictor nn = new NNPredictor(posTrainProfileList, negTrainProfileList);

        String predictorName = "NN BS";
        nn.train();

        List posTestProfileList = dl.getPosTestProfileList();
        List negTestProfileList = dl.getNegTestProfileList();

        List predictionList = nn.predict(posTestProfileList, negTestProfileList);
        Evaluation eval = new Evaluation(predictionList);

        List rocAUCList = new ArrayList();
        rocAUCList.add(eval.getROCAUC());
        List prAUCList = new ArrayList();
        prAUCList.add(eval.getPRAUC());
        List aucLabelList = new ArrayList();
        aucLabelList.add(predictorName);
        Instances inst = eval.getCurve(1);
        List instList = new ArrayList();
        instList.add(inst);

        System.out.println("=== Summary " +testName+ " ("+predictorName+") ===");
        System.out.println(eval.toString());

        plotCurves(instList,rocAUCList, prAUCList, aucLabelList,predictorName + " (" +testName+")");
        
    }

    public HashMap kFoldCrossValidation(ValidationParameters validParams)
    {
        return new HashMap();
    }

    private void leaveOutPeptideTrain(String leaveOutPeptide)
    {
        System.out.println("\tTraining...");

        Chen16FeatureEncoding enc = new Chen16FeatureEncoding();
        posTrainIntSeqList = new ArrayList();
        negTrainIntSeqList = new ArrayList();

        for (int i = 0;i < posTrainProfileList.size();i++)
        {
            ProteinProfile posProfile = (ProteinProfile)posTrainProfileList.get(i);
            String domainSeq = posProfile.getDomainSequence();

            if (encoding == ContactMapSVMPredictor.CHEN16)
            {
                String organismLong = posProfile.getOrganism();
                String organism = PDZSVMUtils.organismLongToShortForm(organismLong);
                domainSeq = enc.getFeatures(domainSeq, organism);
            }
            Collection seqCollection = posProfile.getSequenceMap();
            List seqList  = new ArrayList(seqCollection);
            for (int j = 0; j < seqList.size();j++)
            {
                Sequence seq = (Sequence)seqList.get(j);
                String peptideSeq = seq.seqString();
                if (!peptideSeq.equals(leaveOutPeptide))
                {
                    String posIntString = domainSeq + peptideSeq;
                    posTrainIntSeqList.add(posIntString);
                }
            }

        }
        for (int i = 0;i < negTrainProfileList.size();i++)
        {
            ProteinProfile negProfile = (ProteinProfile)negTrainProfileList.get(i);
            String domainSeq = negProfile.getDomainSequence();
            String organismLong = negProfile.getOrganism();
            String organism = PDZSVMUtils.organismLongToShortForm(organismLong);
            domainSeq = enc.getFeatures(domainSeq, organism);

            Collection seqCollection = negProfile.getSequenceMap();
            List seqList  = new ArrayList(seqCollection);
            for (int j = 0; j < seqList.size();j++)
            {
                Sequence seq = (Sequence)seqList.get(j);
                String peptideSeq = seq.seqString();
                if (!peptideSeq.equals(leaveOutPeptide))
                {
                    String negIntString = domainSeq + peptideSeq;
                    negTrainIntSeqList.add(negIntString);
                }
            }
        }
    }
    private List leaveOutPeptidePredict(List posTestProfileList, List negTestProfileList, String leaveOutPeptide)
    {
        // Clear prediction list!
        predictionList = new ArrayList();
        Data testData = new Data();
        testData.addRawData(posTestProfileList, Constants.CLASS_YES);
        if (!negTestProfileList.isEmpty() && negTestProfileList !=  null)
            testData.addRawData(negTestProfileList,Constants.CLASS_NO);
        Chen16FeatureEncoding enc = new Chen16FeatureEncoding();

        List balPosTestProfileList = new ArrayList();
        List balNegTestProfileList = new ArrayList();
        for (int i =0; i < negTestProfileList.size();i++)
        {
            ProteinProfile negProfile = (ProteinProfile)negTestProfileList.get(i);
            ProteinProfile posProfile = null;
            // get corresponding pos profile
            for (int ii = 0; ii < posTestProfileList.size();ii++)
            {
                posProfile = (ProteinProfile) posTestProfileList.get(ii);
                if (posProfile.getName().equals(negProfile.getName()) &&
                        posProfile.getOrganism().equals(negProfile.getOrganism()))
                break;
            }
            if (posProfile == null)
            {
                balNegTestProfileList.add(negProfile);
                balPosTestProfileList.add(null);
            }
            else
            {
                balNegTestProfileList.add(negProfile);
                balPosTestProfileList.add(posProfile);
            }
        }
        for (int i =0; i < posTestProfileList.size();i++)
        {
            ProteinProfile posProfile = (ProteinProfile)posTestProfileList.get(i);
            ProteinProfile negProfile = null;
            // get corresponding neg profile
            for (int ii = 0; ii < negTestProfileList.size();ii++)
            {
                negProfile = (ProteinProfile) posTestProfileList.get(ii);
                if (negProfile.getName().equals(posProfile.getName()) &&
                        negProfile.getOrganism().equals(posProfile.getOrganism()))
                break;
            }

            if (negProfile == null)
            {
                balNegTestProfileList.add(null);
                balPosTestProfileList.add(posProfile);
            }

        }
        ProteinProfile profile;
        int numPos = 0;
        int numNeg = 0;

        for (int i =0; i < balPosTestProfileList.size();i++)
        {
            ProteinProfile testPosProfile = (ProteinProfile)balPosTestProfileList.get(i);
            ProteinProfile testNegProfile = (ProteinProfile)balNegTestProfileList.get(i);

            if (testPosProfile !=null)
                profile = testPosProfile;
            else
                profile = testNegProfile;

            String organismLong = profile.getOrganism();
            String organism = PDZSVMUtils.organismLongToShortForm(organismLong);
            String name = profile.getName();
            String methodLong = profile.getExperimentalMethod();
            String method = PDZSVMUtils.methodLongToShortForm(methodLong);
            String domainSeqFull = profile.getDomainSequence();

            String domainSeq = enc.getFeatures(domainSeqFull, organism);

            if (testPosProfile!=null)
            {
                Collection seqCollection = testPosProfile.getSequenceMap();
                List seqList  = new ArrayList(seqCollection);
                for (int j = 0; j < seqList.size();j++)
                {

                    Prediction pred;
                    Sequence seq = (Sequence)seqList.get(j);
                    String peptideSeq = seq.seqString();

                    if (!peptideSeq.equals(leaveOutPeptide))
                        continue;
                    numPos = numPos+1;
                    String intSeqString = domainSeq + peptideSeq;

                    // Find NN neighbour interaction in pos or neg int seq lists
                    NNInfo posNN = NN.getNNBindingSiteSeqInfo(intSeqString, posTrainIntSeqList);
                    NNInfo negNN= NN.getNNBindingSiteSeqInfo(intSeqString, negTrainIntSeqList);


                    double decValue = posNN.sim - negNN.sim;
                    // True positive
                    if (decValue >0 )
                    {
                        pred  = new Prediction(1.0,1,decValue, name, domainSeqFull, peptideSeq, organism, method);
                    }
                    else
                    {
                        pred  = new Prediction(0.0,1,decValue, name,domainSeqFull, peptideSeq, organism, method );
                    }
                    predictionList.add(pred);

                }
            }
            if (testNegProfile != null)
            {
                Collection seqCollection = testNegProfile.getSequenceMap();
                List seqList  = new ArrayList(seqCollection);
                for (int j = 0; j < seqList.size();j++)
                {

                    Prediction pred;
                    Sequence seq = (Sequence)seqList.get(j);
                    String peptideSeq = seq.seqString();

                    if (!peptideSeq.equals(leaveOutPeptide))
                        continue;
                    numNeg = numNeg +1;
                    String intSeqString = domainSeq + peptideSeq;
                    // Find NN neighbour interaction in pos or neg int seq lists
                    NNInfo posNN= NN.getNNBindingSiteSeqInfo(intSeqString, posTrainIntSeqList);
                    NNInfo negNN= NN.getNNBindingSiteSeqInfo(intSeqString, negTrainIntSeqList);

                    double decValue = posNN.sim - negNN.sim;

                    // True negative
                    if (decValue<=0)
                    {
                        pred  = new Prediction(0.0,0,decValue, name,domainSeqFull, peptideSeq, organism, method );
                    }
                    else
                    {
                        pred  = new Prediction(1.0,0,decValue, name,domainSeqFull, peptideSeq, organism, method );
                    }

                    predictionList.add(pred);

                }

            }
        }
        if (numPos ==0 || numNeg ==0)
        {
            System.out.println("\tNo positive or negative test data. Skipping...");
            return new ArrayList();
        }
        Evaluation eval = new Evaluation(predictionList);
        double rocAUC = eval.getROCAUC();
        if (rocAUC<0.5)
        {
            System.out.println("\tFlipping signs...");

            for (int ii = 0; ii < predictionList.size();ii++)
            {
                Prediction prediction = (Prediction)predictionList.get(ii);
                double decValue = prediction.getDecValue();
                prediction.setDecValue(-decValue);
            }
        }
        return predictionList;
    }
    public HashMap leaveOutPeptideCrossValidation(ValidationParameters validParams)
    {
        HashMap kFoldResultMap = new HashMap();
        Data trainData = new Data();
        trainData.addRawData(posTrainProfileList, Constants.CLASS_YES);
        trainData.addRawData(negTrainProfileList, Constants.CLASS_NO);
        HashMap peptideRawToNumMap = trainData.getPeptideRawToNumMap();
        Set keys = peptideRawToNumMap.keySet();
        List keyList = new ArrayList(keys);
        StringBuffer seqMismatchPerformanceOutput = new StringBuffer();
        System.out.println("\tNum Folds: " + keyList.size());

        for (int k = 0; k < keyList.size();k++)
        {
            Features ft = (Features)keyList.get(k);
            String leaveOutPeptide = ft.toUndelimitedString();
            System.out.println("\t=== Fold " + k + " ("+leaveOutPeptide+") ===");

            leaveOutPeptideTrain( leaveOutPeptide);
            List foldPredictions = leaveOutPeptidePredict(posTrainProfileList, negTrainProfileList,  leaveOutPeptide);
            if (foldPredictions.size()==0)
                continue;
            Evaluation eval = new Evaluation(foldPredictions);
            System.out.println("\t"+ eval.toString());
            kFoldResultMap.put(k,foldPredictions);
            List trainPeptideSeqList = new ArrayList();
            for (int i=0; i < keyList.size();i++)
            {
                Features ft2 = (Features)keyList.get(i);
                String peptide = ft2.toUndelimitedString();
                if (!peptide.equals(leaveOutPeptide))
                    trainPeptideSeqList.add(peptide);
            }

            seqMismatchPerformanceOutput.append(ValidationFoldOutput.byPeptide(foldPredictions, trainPeptideSeqList));

        }
        if (validParams.p==0)
        {
            try
            {

                String fileName = validParams.predictorName + "_LOOV_ByPeptide.txt";
                System.out.println("Output to: " + validParams.outputDir+"/"+fileName);
                BufferedWriter bw = new BufferedWriter(new FileWriter(new File(validParams.outputDir+"/"+fileName)));
                bw.write(seqMismatchPerformanceOutput.toString());
                bw.close();
            }
            catch(Exception e)
            {
                System.out.println("Exception: " + e);
            }
        }


        return kFoldResultMap;
    }

    public HashMap leaveOutCrossValidation(ValidationParameters validParams)
    {
        if (validParams.type == ValidationParameters.DOMAIN)
        {
            return leaveOutDomainCrossValidation(validParams);
        }

        if (validParams.type==ValidationParameters.PEPTIDE)
        {
            return leaveOutPeptideCrossValidation(validParams);

        }
        else return null;
    }
    public HashMap leaveOutDomainCrossValidation(ValidationParameters validParams)
    {

        HashMap kFoldResultMap = new HashMap();
        HashMap negTrainProfileMap = PDZSVMUtils.profileListToHashMap(negTrainProfileList);
        int k = 0;
        String seqMismatchPerformanceOutput="";
        for (int i=0; i < posTrainProfileList.size();i++)
        {
            ProteinProfile posTestProfile = (ProteinProfile)posTrainProfileList.get(i);
            ProteinProfile negTestProfile = (ProteinProfile)negTrainProfileMap.get(posTestProfile.getName());
            // Need both pos and neg for AUC calculation
            if (negTestProfile == null)
                continue;

            List posTestFoldProfileList = new ArrayList();
            posTestFoldProfileList.add(posTestProfile);
            List negTestFoldProfileList = new ArrayList();
            negTestFoldProfileList.add(negTestProfile);

            List posTrainFoldProfileList= new ArrayList();
            List negTrainFoldProfileList= new ArrayList();
            int numFoldTrainProfiles = 0;

            for (int j = 0; j< posTrainProfileList.size();j++)
            {
                ProteinProfile posTrainProfile = (ProteinProfile)posTrainProfileList.get(j);
                String expMethodTrain = posTrainProfile.getExperimentalMethod();
                String expMethodTest = posTestProfile.getExperimentalMethod();

                if (!posTrainProfile.getName().equals(posTestProfile.getName()) )
                {
                    numFoldTrainProfiles = numFoldTrainProfiles +1;
                    posTrainFoldProfileList.add(posTrainProfile);

                    ProteinProfile negTrainProfile = (ProteinProfile)negTrainProfileMap.get(posTrainProfile.getName());
                    negTrainFoldProfileList.add(negTrainProfile);
                }
                else
                {
                    if(!expMethodTrain.equals(expMethodTest))
                    {   numFoldTrainProfiles = numFoldTrainProfiles +1;
                        posTrainFoldProfileList.add(posTrainProfile);

                        ProteinProfile negTrainProfile = (ProteinProfile)negTrainProfileMap.get(posTrainProfile.getName());

                        negTrainFoldProfileList.add(negTrainProfile);
                    }
                    else
                    {
                        System.out.println("\tNames are equal: " + posTrainProfile.getName() + "\t" + posTestProfile.getName());
                    }
                }
            }
            //System.out.println("\t" + numFoldTrainProfiles);
            //System.out.println("\t" + posTrainFoldProfileList.size());

            //System.out.println("\t" + negTrainFoldProfileList.size());

            System.out.println("\t=== Fold " + k + " ===");
            NNPredictor nnFold = new NNPredictor(posTrainFoldProfileList, negTrainFoldProfileList);
            nnFold.train();
            List foldPredictions = nnFold.predict(posTestFoldProfileList, negTestFoldProfileList);
            Evaluation eval = new Evaluation(foldPredictions);
            System.out.println("\t"+ eval.toString());
            kFoldResultMap.put(k,foldPredictions);
            k = k+1;
            ValidationFoldOutput ss = new ValidationFoldOutput(posTrainFoldProfileList, negTrainFoldProfileList);
            seqMismatchPerformanceOutput = seqMismatchPerformanceOutput + ss.byDomain(foldPredictions);

        }
        if (validParams.d==0)
        {
            try
            {
                String fileName = validParams.predictorName + "_LOOV_ByDomain.txt";
                System.out.println("Output to: " + validParams.outputDir+"/"+fileName);
                BufferedWriter bw = new BufferedWriter(new FileWriter(new File(validParams.outputDir+"/"+fileName)));
                bw.write(seqMismatchPerformanceOutput.toString());
                bw.close();
            }
            catch(Exception e)
            {
                System.out.println("Exception: " + e);
            }
        }
        return kFoldResultMap;
    }

}
