package org.baderlab.pdzsvm.analysis;

import org.baderlab.brain.*;

import java.util.*;
import java.util.List;
import java.io.*;
import java.awt.image.BufferedImage;
import java.awt.*;
import com.lowagie.text.Font;

import org.baderlab.pdzsvm.utils.PDZSVMUtils;
import org.baderlab.pdzsvm.utils.Constants;
import org.baderlab.pdzsvm.data.manager.DataFileManager;
import org.baderlab.pdzsvm.data.DataLoader;
import org.baderlab.pdzsvm.predictor.nn.NN;
import org.baderlab.pdzsvm.predictor.pwm.PWMPredictor;
import org.baderlab.pdzsvm.predictor.pwm.PWM;
import org.biojava.bio.seq.db.HashSequenceDB;
import org.biojava.bio.seq.Sequence;
import org.biojava.bio.seq.ProteinTools;
import org.biojava.bio.dp.SimpleWeightMatrix;
import weka.core.Utils;
import com.lowagie.text.*;
import com.lowagie.text.pdf.PdfWriter;
import com.lowagie.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfPCell;


/**
 * Copyright (c) 2010 University of Toronto
 * Code written by: Shirley Hui
 * Authors: Shirley Hui, Gary Bader
 *
 * This file is part of PDZSVM.
 *
 * PDZSVM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PDZSVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * documentation provided hereunder is on an "as is" basis, and the
 * University of Toronto has no obligations to provide maintenance,
 * support, updates, enhancements or modifications.  In no event shall
 * the University of Toronto be liable to any party for direct, indirect,
 * special, incidental or consequential damages, including lost profits,
 * arising out of the use of this software and its documentation, even if
 * the University of Toronto has been advised of the possibility of such
 * damage. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PDZSVM.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Compare predicted and experimentally determines sequence logos representing
 * domain binding specificity.
 */
public class CompareProfiles {
    private static final String parentDir = "/Tables";
    private static final String dirName = "/LogoTables";
    private List allTrainProfileList;

    private CompareProfiles()
    {
        DataLoader dl = new DataLoader();
        dl.loadMouseChenTrain();
        dl.loadHumanTrain(Constants.PWM);
        List posTrainProfileList = dl.getPosTrainProfileList();
        List negTrainProfileList = dl.getNegTrainProfileList();

        allTrainProfileList = PDZSVMUtils.allProfileList( posTrainProfileList,  negTrainProfileList);

    }
    // Utility method to double check NN for scanned domains.  This relies on a project file pointing to the
    // locations of the svm predicted peptide files (i.e. DOMAIN_NAME.svm.pep.txt).  Peptide files are
    // outputted from ProteomeScan (validate?=Y)
    private void printNN(String organism)
    {
        String projectExpFileName  = "";
        if (organism.equals("Fly"))
            projectExpFileName = DataFileManager.OUTPUT_ROOT_DIR +"/ProteomeScan/Fly/projectFly-scan.txt";
        else if (organism.equals("Worm"))
            projectExpFileName = DataFileManager.OUTPUT_ROOT_DIR +"/ProteomeScan/Worm/projectWorm-scan.txt";
        else
            projectExpFileName = DataFileManager.OUTPUT_ROOT_DIR +"/ProteomeScan/Human/projectHuman-scan.txt";

        List profileExpList = PDZSVMUtils.readProteinProfileList(projectExpFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, 5, true);
        for (int i=0; i < profileExpList.size();i++)
        {
            ProteinProfile profile = (ProteinProfile)profileExpList.get(i);
            String domainSeq = profile.getDomainSequence();
            String organismLong = profile.getOrganism();
            String organismShort = PDZSVMUtils.organismLongToShortForm(organismLong);

            ProteinProfile nnProfile = NN.getNNBindingSiteSeqProfile(domainSeq, organismShort, allTrainProfileList);

            double nnSim = NN.getIdentity(profile, nnProfile);
            System.out.println(profile.getName() +"\t" + nnProfile.getName() + "\t" + nnProfile.getOrganism()+"\t" + nnSim +"\t"+ nnProfile.getNumSequences());
        }


    }
    public static void main(String[] args)
    {
        CompareProfiles c = new CompareProfiles();
        c.compareHuman();
        //c.compare();
        //c.makePhageGenomicLogo();
    }

    public void makeHeading(PdfPTable table)
    {
        int pad = 4;
        Font font = new Font(Font.HELVETICA, 8f, Font.NORMAL, Color.BLACK);
        try
        {
            PdfPCell cell  = new PdfPCell(new Phrase("Domain Name",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("NN Sim",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("Experiment",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("SVM Predicted",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("Profile Sim",font));
            cell.setPadding(pad);
            table.addCell(cell);
        }catch(Exception e)
        {
            System.out.println("Exception: " + e);
            e.printStackTrace();
        }

    }
    public void makeHumanHeading(PdfPTable table)
    {
        int pad = 4;
        Font font = new Font(Font.HELVETICA, 8f, Font.NORMAL, Color.BLACK);
        try
        {
            PdfPCell cell  = new PdfPCell(new Phrase("Domain Name",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("NN Sim",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("Optimal",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("Genomic",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("SVM Predicted",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("Optimal Profile Sim",font));
            cell.setPadding(pad);
            table.addCell(cell);

            cell = new PdfPCell(new Phrase("Genomic Profile Sim",font));
            cell.setPadding(pad);
            table.addCell(cell);
        }catch(Exception e)
        {
            System.out.println("Exception: " + e);
            e.printStackTrace();
        }

    }
    public void compareHuman()
    {

        float[] colsWidth = {2f,1.15f,2.4f,2.4f,2.4f,2f,2f}; // Code 1

        //PdfPTable bigTable  = new PdfPTable(1); // Code 1
        //bigTable.getDefaultCell().setBorderWidth(0);
        List profileSVMList;
        List profileHumanExpList = new ArrayList();
        List profileHumanGenExpList = new ArrayList();

        String projectSVMFileName = "";
        String projectExpFileName = "";
        String projectExpGenFileName = "";


        projectSVMFileName = DataFileManager.OUTPUT_ROOT_DIR+"/ProteomeScan/Human/projectHuman-scan.txt";
        projectExpFileName = DataFileManager.DATA_ROOT_DIR+"/Data/Human/PDZ/ProjectFiles/projectFileHuman-allgenomic.txt";
        projectExpGenFileName = "/Users/shirleyhui/Desktop/Results/Logos/projectFileHumanPhageGenomic.txt";

        profileSVMList = PDZSVMUtils.readProteinProfileList(projectSVMFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true);
        profileHumanExpList = PDZSVMUtils.readProteinProfileList(projectExpFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true);
        profileHumanGenExpList = PDZSVMUtils.readProteinProfileList(projectExpGenFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true);
        int numPDExpSeq = 10;


        int pad = 4;
        Font font = new Font(Font.HELVETICA, 8f, Font.NORMAL, Color.BLACK);
        try
        {
            PdfPTable table = new PdfPTable(colsWidth); // Code 1
            table.setWidthPercentage(60);
            makeHumanHeading(table);
            int added = 0;
            for (int i =0; i < profileSVMList.size();i++)
            {

                ProteinProfile svmProfile = (ProteinProfile) profileSVMList.get(i);
                String domainName = svmProfile.getName();

                String domainSeq = svmProfile.getDomainSequence();
                String organismLong = svmProfile.getOrganism();
                String organismShort = PDZSVMUtils.organismLongToShortForm(organismLong);
                // Get corresponding expProfile
                String org = "";
                ProteinProfile expProfile= null;
                ProteinProfile expGenProfile = null;
                if (svmProfile.getOrganism().equals(Constants.HUMAN_STRING))

                {   boolean found = false;
                    org = "Human";
                    for (int ii=0; ii < profileHumanExpList.size();ii++)
                    {
                        expProfile   = (ProteinProfile)profileHumanExpList.get(ii);
                        for (int kk=0;kk < profileHumanGenExpList.size();kk++)
                        {
                            expGenProfile   = (ProteinProfile)profileHumanGenExpList.get(kk);
                            if (expProfile.getName().equals(expGenProfile.getName()))
                                break;
                        }
                        if (expProfile.getName().equals(svmProfile.getName()))
                        {
                            
                            found = true;
                            break;
                        }
                    }
                    if (!found) expProfile = null;
                }

                String profileSim = "0.0";
                String profileGenSim = "0.0";
                String name = svmProfile.getName() +"\n"+org;
                com.lowagie.text.Image expImage= null;
                com.lowagie.text.Image expGenImage = null;
                if (expProfile==null)
                {
                    //System.out.println("Exp profile for " + svmProfile.getName() + " is null...");
                    continue;
                }
                else
                {
                    //System.out.println("\tProcessing: " + domainName +  " with " +expProfile.getName());
                    int numExpSeq = numPDExpSeq;
                    profileSim= Utils.doubleToString(1.0-ProteinProfileDistance.calculateDistributionDistance(svmProfile, expProfile),3);
                    profileGenSim= Utils.doubleToString(1.0-ProteinProfileDistance.calculateDistributionDistance(svmProfile, expGenProfile),3);


                    if (expProfile.getNumSequences() < numExpSeq)
                    {
                        //System.out.println("\tNot enough sequences...");
                        continue;
                    }
                    ProteinSequenceLogo expLogo = new ProteinSequenceLogo( expProfile, 240);
                    expLogo.sequenceLogoSetStartIndex(-4);
                    BufferedImage expImgLogo = expLogo.drawSequenceLogo();
                    expImage = com.lowagie.text.Image.getInstance(expImgLogo, null);

                    ProteinSequenceLogo expGenLogo = new ProteinSequenceLogo( expGenProfile, 240);
                    expGenLogo.sequenceLogoSetStartIndex(-4);
                    BufferedImage expGenImgLogo = expGenLogo.drawSequenceLogo();
                    expGenImage = com.lowagie.text.Image.getInstance(expGenImgLogo, null);
                    //System.out.println("\tMade exp profile image...");

                }
                ProteinSequenceLogo svmLogo = new ProteinSequenceLogo( svmProfile, 240);

                svmLogo.sequenceLogoSetStartIndex(-4);
                BufferedImage svmImgLogo = svmLogo.drawSequenceLogo();
                com.lowagie.text.Image svmImage = com.lowagie.text.Image.getInstance(svmImgLogo, null);

                ProteinProfile nnProfile = NN.getNNBindingSiteSeqProfile(domainSeq, organismShort, allTrainProfileList);
                double nnSim = NN.getIdentity(svmProfile, nnProfile);

                //System.out.println("\tMade svm profile image...");
                System.out.println("\tAdding row: "+domainName +" -> "+nnSim+" -> expImage -> expGenImage -> svnImage -> "+ profileSim + "->" + profileGenSim);
                added = added+1;
                PdfPCell cell = new PdfPCell(new Phrase(name,font));
                cell.setPadding(pad);

                if (table==null)
                    System.out.println("Table is null");
                table.addCell(cell);

                cell = new PdfPCell(new Phrase(Utils.doubleToString(nnSim,2,2),font));
                cell.setPadding(pad);

                table.addCell(cell);

                if (expImage!=null)
                {
                    table.addCell(expImage);
                    table.addCell(expGenImage);

                }
                else
                    table.addCell("No genomic profile available.");

                table.addCell(svmImage);

                cell = new PdfPCell(new Phrase(profileSim,font));

                cell.setPadding(pad);
                table.addCell(cell);

                cell = new PdfPCell(new Phrase(profileGenSim,font));

                cell.setPadding(pad);
                table.addCell(cell);

            }
            //bigTable.addCell(table);


            Document document = new Document();

            PdfWriter.getInstance(document,
                    new FileOutputStream(DataFileManager.OUTPUT_ROOT_DIR+parentDir+dirName+"/CompareLogos-Gen.pdf"));
            document.open();

            document.add(table);

            document.close();

        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
            e.printStackTrace();
        }

    }
    public void compare()
    {

        float[] colsWidth = {2f,1.15f,2.4f,2.4f,1.5f}; // Code 1

        PdfPTable bigTable  = new PdfPTable(2); // Code 1
        bigTable.getDefaultCell().setBorderWidth(0);
        List profileSVMList;
        List profileHumanExpList = new ArrayList();

        List profileWormExpList = new ArrayList();
        List profileFlyExpList = new ArrayList();

        String projectSVMFileName = "";
        String projectExpFileName = "";


        projectSVMFileName = DataFileManager.OUTPUT_ROOT_DIR+"/ProteomeScan/Human/projectHuman-scan.txt";
        projectExpFileName = DataFileManager.DATA_ROOT_DIR+"/Data/Human/PDZ/ProjectFiles/projectFileHuman-allgenomic.txt";

        profileSVMList = PDZSVMUtils.readProteinProfileList(projectSVMFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true);
        profileHumanExpList = PDZSVMUtils.readProteinProfileList(projectExpFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true);

        projectSVMFileName = DataFileManager.OUTPUT_ROOT_DIR+"/ProteomeScan/Worm/projectWorm-scan.txt";
        projectExpFileName = DataFileManager.DATA_ROOT_DIR+"/Data/Worm/PDZ/ProjectFiles/projectFile-pos-chen-worm.txt";
        int numPDExpSeq = 10;

        profileSVMList.addAll(PDZSVMUtils.readProteinProfileList(projectSVMFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true));
        profileWormExpList = PDZSVMUtils.readProteinProfileList(projectExpFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true);

        projectSVMFileName = DataFileManager.OUTPUT_ROOT_DIR+"/ProteomeScan/Fly/projectFly-scan.txt";
        projectExpFileName = DataFileManager.DATA_ROOT_DIR+"/Data/Fly/PDZ/ProjectFiles/projectFile-pos-chen-fly.txt";
        int numPMExpSeq= 5;
        profileSVMList.addAll(PDZSVMUtils.readProteinProfileList(projectSVMFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true));
        profileFlyExpList = PDZSVMUtils.readProteinProfileList(projectExpFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true);

        int pad = 4;
        Font font = new Font(Font.HELVETICA, 8f, Font.NORMAL, Color.BLACK);
        try
        {
            PdfPTable table = new PdfPTable(colsWidth); // Code 1
            table.setWidthPercentage(45);
            makeHeading(table);
            int added = 0;
            for (int i =0; i < profileSVMList.size();i++)
            {
                if (added==6)
                {
                    bigTable.addCell(table);
                    table = new PdfPTable(colsWidth); // Code 1
                    table.setWidthPercentage(45);
                    makeHeading(table);

                }
                ProteinProfile svmProfile = (ProteinProfile) profileSVMList.get(i);
                String domainName = svmProfile.getName();
                if (domainName.equals("NAB1-1") )
                    continue;

                String domainSeq = svmProfile.getDomainSequence();
                String organismLong = svmProfile.getOrganism();
                String organismShort = PDZSVMUtils.organismLongToShortForm(organismLong);
                // Get corresponding expProfile
                String org = "";
                ProteinProfile expProfile= null;
                if (svmProfile.getOrganism().equals(Constants.HUMAN_STRING))

                {   boolean found = false;
                    org = "Human";
                    for (int ii=0; ii < profileHumanExpList.size();ii++)
                    {
                        expProfile   = (ProteinProfile)profileHumanExpList.get(ii);
                        if (expProfile.getName().equals(svmProfile.getName()))
                        {
                            found = true;
                            break;
                        }
                    }
                    if (!found) expProfile = null;
                }
                else if (svmProfile.getOrganism().equals(Constants.WORM_STRING))

                {
                    boolean found = false;
                    org = "Worm";

                    for (int ii=0; ii < profileWormExpList.size();ii++)
                    {
                        expProfile   = (ProteinProfile)profileWormExpList.get(ii);
                        if (expProfile.getName().equals(svmProfile.getName()))
                        {
                            found = true;
                            break;
                        }
                    }
                    if (!found) expProfile = null;
                }
                else
                {
                    boolean found = false;
                    org = "Fly";

                    for (int ii=0; ii < profileFlyExpList.size();ii++)
                    {
                        expProfile   = (ProteinProfile)profileFlyExpList.get(ii);
                        if (expProfile.getName().equals(svmProfile.getName()))
                        {
                            found = true;
                            break;
                        }
                    }
                    if (!found) expProfile = null;
                }
                String profileSim = "0.0";
                String name = svmProfile.getName() +"\n"+org;
                com.lowagie.text.Image expImage= null;
                if (expProfile==null)
                {
                    //System.out.println("Exp profile for " + svmProfile.getName() + " is null...");
                    continue;
                }
                else
                {
                    //System.out.println("\tProcessing: " + domainName +  " with " +expProfile.getName());
                    int numExpSeq = numPMExpSeq;
                    profileSim= Utils.doubleToString(1.0-ProteinProfileDistance.calculateDistributionDistance(svmProfile, expProfile),3);
                    if (expProfile.getExperimentalMethod().equals(Constants.PHAGE_DISPLAY_HIGH_STRING) ||
                            expProfile.getExperimentalMethod().equals(Constants.PHAGE_DISPLAY_LOW_STRING))
                    {
                        numExpSeq= numPDExpSeq;
                    }

                    if (expProfile.getNumSequences() < numExpSeq)
                    {
                        //System.out.println("\tNot enough sequences...");
                        continue;
                    }
                    ProteinSequenceLogo expLogo = new ProteinSequenceLogo( expProfile, 240);
                    expLogo.sequenceLogoSetStartIndex(-4);
                    BufferedImage expImgLogo = expLogo.drawSequenceLogo();
                    expImage = com.lowagie.text.Image.getInstance(expImgLogo, null);
                    //System.out.println("\tMade exp profile image...");

                }
                ProteinSequenceLogo svmLogo = new ProteinSequenceLogo( svmProfile, 240);

                svmLogo.sequenceLogoSetStartIndex(-4);
                BufferedImage svmImgLogo = svmLogo.drawSequenceLogo();
                com.lowagie.text.Image svmImage = com.lowagie.text.Image.getInstance(svmImgLogo, null);

                ProteinProfile nnProfile = NN.getNNBindingSiteSeqProfile(domainSeq, organismShort, allTrainProfileList);
                double nnSim = NN.getIdentity(svmProfile, nnProfile);

                //System.out.println("\tMade svm profile image...");
                System.out.println("\tAdding row: "+domainName +" -> "+nnSim+" -> expImage -> svmImage -> "+ profileSim);
                added = added+1;
                PdfPCell cell = new PdfPCell(new Phrase(name,font));
                cell.setPadding(pad);

                if (table==null)
                    System.out.println("Table is null");
                table.addCell(cell);

                cell = new PdfPCell(new Phrase(Utils.doubleToString(nnSim,2,2),font));
                cell.setPadding(pad);

                table.addCell(cell);

                if (expImage!=null)
                    table.addCell(expImage);
                else
                    table.addCell("No genomic profile available.");

                table.addCell(svmImage);

                cell = new PdfPCell(new Phrase(profileSim,font));

                cell.setPadding(pad);
                table.addCell(cell);

            }
            bigTable.addCell(table);


            Document document = new Document();

            PdfWriter.getInstance(document,
                    new FileOutputStream(DataFileManager.OUTPUT_ROOT_DIR+parentDir+dirName+"/CompareLogos.pdf"));
            document.open();

            document.add(bigTable);

            document.close();

        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
            e.printStackTrace();
        }

    }
    public void makePhageGenomicLogo()
    {
        String projectExpFileName = DataFileManager.DATA_ROOT_DIR+"/Data/Human/PDZ/ProjectFiles/projectFileHuman-allgenomic.txt";
        List profileHumanExpList = PDZSVMUtils.readProteinProfileList(projectExpFileName, DataFileManager.NNK_CODON_BIAS_FILENAME, Constants.NUM_RES_POS, true);
        HashMap genomeSeqToGeneNameMap = new HashMap();
        String fastaFileName = DataFileManager.DATA_ROOT_DIR+"/Data/Ensembl/Homo_sapiens.GRCh37.56.pep.all.fa";
        try
        {
            genomeSeqToGeneNameMap=  PDZSVMUtils.genomeSeqToGeneNameMap( fastaFileName, 5);

        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
        }

        Collection keys = genomeSeqToGeneNameMap.keySet();
        List genomeSeqList = new ArrayList(keys);

        HashSequenceDB genomeSeqDB = new HashSequenceDB();
        int numAdded = 0;
        int numPos = 5;
        for (int i=0; i < genomeSeqList.size();i++)
        {

            try
            {
                String seq = (String) genomeSeqList.get(i);
                seq = seq.substring(seq.length()-numPos, seq.length());
                Sequence prot = ProteinTools.createProteinSequence(seq, "Peptide" + i);
                genomeSeqDB.addSequence(prot);
                numAdded = numAdded+1;

            }
            catch(Exception e)
            {
                System.out.println("Exception: " + e);
            } //
        }
        System.out.println("\tGenomic seq size: " + numAdded);

        List[] sortedLists = new List[2];
        for (int i=0; i< profileHumanExpList.size();i++)
        {
            ProteinProfile profile = (ProteinProfile)profileHumanExpList.get(i);
            PWM pwm = new PWM(profile);
            sortedLists = getSortedLists(genomeSeqList, pwm);

            List sortedSeqList = sortedLists[0];
            List sortedPWMScoreList = sortedLists[1];
            int numTopPercent = (int)Math.rint(sortedPWMScoreList.size()*0.01);
            if (numTopPercent == 0)
                numTopPercent =1;
            //System.out.println("Num top %: " + numTopPercent);

            List predPosSeqList = sortedSeqList.subList(0,numTopPercent);
            HashSequenceDB seqDB = new HashSequenceDB();
            int peptideNum = 0;
            for (int j=0; j< predPosSeqList.size();j++)
            {
                String seqString = (String)predPosSeqList.get(j);

                try
                {
                    Sequence seq = ProteinTools.createProteinSequence(seqString,"Peptide" + peptideNum);
                    seqDB.addSequence(seq);
                    peptideNum = peptideNum+1;
                }
                catch(Exception e)
                {
                    System.out.println("Exception" +e);

                }
            }
            String outDir = DataFileManager.OUTPUT_ROOT_DIR + "/Logos";
            ProteinProfile phageGenomicProfile = null;
            if (seqDB.ids().size()>0)
            {
                phageGenomicProfile = PDZSVMUtils.makeProfile(profile.getName(), profile.getDomainNumber(), profile.getDomainSequence(), profile.getOrganism(), seqDB);
                //PDZSVMUtils.saveAsLogo(phageGenomicProfile,phageGenomicProfile.getName() + "-0Valid",outDir,ClassIndex.YES);
                PDZSVMUtils.saveToProfile(phageGenomicProfile,outDir + "/" +profile.getName()+".PHAGEGEN.pep.txt");

            }
        }
    }
    public List[] getSortedLists(List seqList, PWM pwm)
    {

        double[] pwmScores = new double[seqList.size()];
        for (int j=0;j<seqList.size();j++)
        {
            String seqString = (String)seqList.get(j);
            double score = pwm.score(seqString);
            pwmScores[j] = score;

        }

        int[] lowToHigh  = Utils.sort(pwmScores);
        int[] highToLo = new int[lowToHigh.length];
        int end = lowToHigh.length-1;

        for (int ii=0; ii<lowToHigh.length;ii++)
        {
            int ixi = lowToHigh[end-ii];
            highToLo[ii] = ixi;
        }
        int[] ix;

        ix = highToLo;
        List sortedSequenceList  = new ArrayList();
        List sortedPWMScoreList  = new ArrayList();
        if (ix.length==1)
        {
            Sequence seq = (Sequence)seqList.get(0);
            String seqString = seq.seqString();
            sortedSequenceList.add(seqString);
            sortedPWMScoreList.add(pwmScores[0]);
        }
        else
        {
            for (int ii=0; ii < ix.length;ii++)
            {
                int ixi = ix[ii];
                String seqString = (String)seqList.get(ixi);

                sortedSequenceList.add(seqString);
                sortedPWMScoreList.add(pwmScores[ixi]);
            }


        }
        List[] sortedLists = new ArrayList[2];
        sortedLists[0] = sortedSequenceList;
        sortedLists[1] = sortedPWMScoreList;
        return sortedLists;
    }
}
