package org.baderlab.pdzsvmstruct.utils;

import org.baderlab.pdzsvmstruct.data.manager.DataFileManager;

import java.util.*;
import java.io.*;

/**
 * Copyright (c) 2011 University of Toronto
 * Code written by: Shirley Hui
 * Authors: Shirley Hui, Gary Bader
 *
 * This file is part of PDZSVMStruct.
 *
 * PDZSVM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PDZSVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * documentation provided hereunder is on an "as is" basis, and the
 * University of Toronto has no obligations to provide maintenance,
 * support, updates, enhancements or modifications.  In no event shall
 * the University of Toronto be liable to any party for direct, indirect,
 * special, incidental or consequential damages, including lost profits,
 * arising out of the use of this software and its documentation, even if
 * the University of Toronto has been advised of the possibility of such
 * damage. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PDZSVMStruct.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * This file handles iRefIndex interactions and is called by ProteomeScan to see if a
 * predicted interaction corresponds to a known protein-protein interaction.
 */
public class IRefIndexInteractome {

    private String organism;
    private String fastaFileName;
    private HashMap ensemblToUniprotMap;
    private HashMap iRefWebInteractomeMap;
    private HashMap proteinNameToUniprotIdMap;
    public IRefIndexInteractome(String organism)
    {
        this.organism = organism;
        loadiRefWebInteratome();
        loadEnsemblToUniProtMap();
        loadProteinNameToUniprotIdMap();
    }

    private void loadProteinNameToUniprotIdMap()
    {
        proteinNameToUniprotIdMap = new HashMap();
        String proteinNameToUniprotIdFileName = DataFileManager.DATA_ROOT_DIR + "/Data/Uniprot/ProteinNameToUniprotIdHuman.txt";
        try
        {
            BufferedReader br = new BufferedReader(new FileReader(new File(proteinNameToUniprotIdFileName)));
            String line = "";
            while((line=br.readLine())!=null)
            {
                String[] splitLine = line.split("\t");
                proteinNameToUniprotIdMap.put(splitLine[0],splitLine[1]);
            }
        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
        }
    }
    public HashMap getInteractions()
    {
        return iRefWebInteractomeMap;
    }
    public List getInteractions(String domainName)
    {
        Set keys = iRefWebInteractomeMap.keySet();
        List keyList = new ArrayList(keys);
        String uniprotId = (String)proteinNameToUniprotIdMap.get(domainName);
        List interactorList = new ArrayList();
        for (int i=0; i < keyList.size();i++)
        {
            HashSet intSet = (HashSet)keyList.get(i);
            List intSetList = new ArrayList(intSet);
            String intA = "";
            String intB = "";
            if (intSetList.size()<2)
            {
                intA = (String)intSetList.get(0);
                intB = (String)intSetList.get(0);
            }
            else
            {
                intA = (String)intSetList.get(0);
                intB = (String)intSetList.get(1);
            }
            if (!intA.equals(uniprotId) && !intB.equals(uniprotId))
                continue;
            if (intA.equals(uniprotId))
                interactorList.add(intB);
            else
                interactorList.add(intA);
        }
        return interactorList;
    }
    public static void main(String[] args)
    {
        IRefIndexInteractome iRefWeb = new IRefIndexInteractome(Constants.HUMAN);
        String domainName = "CAR14";
        int numInteractions = iRefWeb.numiRefWebInteractions(domainName);
        System.out.println(domainName+"," + numInteractions);
        //HashMap map = iRefWeb.getInteractions();
        List intList = iRefWeb.getInteractions(domainName);
        System.out.println(intList.size());
        System.out.println(intList);
    }
    private String loadInteractors(String interactorLine)
    {
        String interactors = "";
        String[] splitLine = interactorLine.split("\\|");
        List uniqueIdList = new ArrayList();

        for (int i=0 ; i < splitLine.length;i++)
        {
            String split = splitLine[i];
            String[] spiltSplitLine = split.split(":");
            for (int j = 0 ; j < spiltSplitLine.length;j++)
            {
                if (spiltSplitLine[0].startsWith("uniprotkb"))
                {
                    String uniprotId = spiltSplitLine[1];
                    int slashIx = uniprotId.indexOf('-');

                    if (slashIx != -1)
                    {
                        uniprotId = uniprotId.substring(0,slashIx);
                    }

                    if (!uniqueIdList.contains(uniprotId))
                    {
                        if (interactors.equals(""))
                            interactors = uniprotId;
                        else
                            interactors = interactors + "|" + uniprotId;

                        uniqueIdList.add(uniprotId);
                    }
                }
            }
        }
        return interactors;
    }
    private void loadInteraction_minitab(String intLine)
    {

        String[] splitLine = intLine.split("\t");
        String intA = splitLine[2];
        String intB = splitLine[3];

        Set intKey = new HashSet();
        intKey.add(intA);
        intKey.add(intB);
        iRefWebInteractomeMap.put(intKey,intKey);

    }
    private void loadInteraction(String intLine)
    {

        String[] splitLine = intLine.split("\t");
        String intALine = splitLine[1];
        String intBLine = splitLine[2];

        String intA = "";
        String intB = "";


        intA = loadInteractors(intALine);
        intB = loadInteractors(intBLine);
        if (intA.equals("") || intB.equals(""))
            return;
        //System.out.println(intA + "\t" + intB);

        String[] intAs = intA.split("\\|");
        String[] intBs = intB.split("\\|");

        for (int i=0; i < intAs.length;i++)
        {
            for (int j = 0; j < intBs.length;j++)
            {
                Set intKey = new HashSet();
                intKey.add(intAs[i]);
                intKey.add(intBs[j]);
                iRefWebInteractomeMap.put(intKey,intKey);
            }
        }
    }

    public int numiRefWebInteractions(String proteinName)
    {
        int numInteractions = 0;

        List uniprotIdAList = proteinNameToUniprotIds(proteinName);

        // parse interactome and count lines in the file that contain uniprotId

        for (int i=0;i < uniprotIdAList.size();i++)
        {
            String uniprotIdA = (String)uniprotIdAList.get(i);

            Set keys = iRefWebInteractomeMap.keySet();
            List keyList = new ArrayList(keys);
            for (int j=0; j < keyList.size();j++)
            {
                HashSet intKey = (HashSet)keyList.get(j);
                if (intKey.contains(uniprotIdA))
                {

                    numInteractions = numInteractions+1;

                }

            }
        }

        return numInteractions;
    }
    public void loadiRefWebInteratome_minitab()
    {
        if (organism.equals(Constants.HUMAN))
            fastaFileName = DataFileManager.DATA_ROOT_DIR+"/Data/iRefWeb/mitab_lite_human.txt";

        iRefWebInteractomeMap = new HashMap();
        try
        {
            BufferedReader br = new BufferedReader(new FileReader(new File(fastaFileName)));
            String line = "";
            int numLines = 0;
            while((line=br.readLine())!=null)
            {
                if (!line.startsWith("#"))
                {
                    // read interactions
                    loadInteraction_minitab(line);
                    numLines = numLines+1;
                }
            }
            br.close();
            //System.out.println("\tNum irefIndex interations read: " + numLines);

        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
            e.printStackTrace();
        }

    }

    public void loadiRefWebInteratome()
    {
        if (organism.equals(Constants.HUMAN))
            fastaFileName = DataFileManager.DATA_ROOT_DIR+"/Data/iRefWeb/interactome_mitab_human_2011.txt";

        iRefWebInteractomeMap = new HashMap();
        try
        {
            BufferedReader br = new BufferedReader(new FileReader(new File(fastaFileName)));
            String line = "";
            int numLines = 0;
            while((line=br.readLine())!=null)
            {
                if (!line.startsWith("#"))
                {
                    // read interactions
                    loadInteraction(line);
                    numLines = numLines+1;
                }
            }
            br.close();
            //System.out.println("\tNum irefIndex interations read: " + numLines);

        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
            e.printStackTrace();
        }

    }
    private List proteinNameToUniprotIds(String proteinName)
    {
        List uniprotIdList = new ArrayList();

        if (organism.equals(Constants.HUMAN))
        {

            String uniprotId = (String)proteinNameToUniprotIdMap.get(proteinName);
            uniprotIdList.add(uniprotId);
        }


        return uniprotIdList;
    }

    public boolean lookupUniprotId(String proteinName, String uniprotIdB)
    {
        boolean match = false;
        //System.out.println("\tLooking up iRefWeb interactions for " + proteinName+"...");
        // One protein can map to more than one uniprot id!
        List uniprotIdAList = proteinNameToUniprotIds(proteinName);
        for (int i=0; i < uniprotIdAList.size();i++)
        {
            // convert ensemblTRSIds to uniprot
            String uniprotIdA = (String)uniprotIdAList.get(i);

            Set lookupSet = new HashSet();
            lookupSet.add(uniprotIdA);
            lookupSet.add(uniprotIdB);

            if (iRefWebInteractomeMap.get(lookupSet) == null)
            {
                //System.out.println(lookupSet.toString() + " not found!");
            }
            else
            {
                match = true;
                //System.out.println(lookupSet.toString() + "  found!");
                break;
            }
        }
        return match;
    }
    public boolean lookupEnsemblId(String proteinName, String ensemblTRSIdB)
    {
        boolean match = false;
        //System.out.println("\tLooking up iRefWeb interactions for " + proteinName+"...");
        // One protein can map to more than one uniprot id!
        List uniprotIdAList = proteinNameToUniprotIds(proteinName);
        for (int i=0; i < uniprotIdAList.size();i++)
        {
            // convert ensemblTRSIds to uniprot
            String uniprotIdA = (String)uniprotIdAList.get(i);
            List uniprotIdBList = (List)ensemblToUniprotMap.get(ensemblTRSIdB);
            if (uniprotIdBList == null)
                continue;
            for (int j=0;j< uniprotIdBList.size();j++)
            {
                String uniprotIdB = (String)uniprotIdBList.get(j);
                //System.out.println(uniprotIdA + "\t" + uniprotIdB);
                Set lookupSet = new HashSet();
                lookupSet.add(uniprotIdA);
                lookupSet.add(uniprotIdB);

                if (iRefWebInteractomeMap.get(lookupSet) == null)
                {
                    //System.out.println(lookupSet.toString() + " not found!");
                }
                else
                {
                    match = true;
                    //System.out.println(lookupSet.toString() + "  found!");
                    break;
                }
            }
        }
        return match;

    }
    public List ensemblToUniprotIds(String ensemblId)
    {
        List uniprotIdList = (List)ensemblToUniprotMap.get(ensemblId);
        return uniprotIdList;
    }
    public void loadEnsemblToUniProtMap()
    {
        ensemblToUniprotMap = new HashMap();
        String ensemblToUniprotSFileName = "";
        try
        {
            if (organism.equals(Constants.HUMAN))
                ensemblToUniprotSFileName =DataFileManager.DATA_ROOT_DIR+"/Data/iRefWeb/EnsemblToUniprotMapping_Human.txt";
            else return;

            BufferedReader br = new BufferedReader(new FileReader(new File(ensemblToUniprotSFileName)));
            String line ="";
            while((line=br.readLine())!=null)
            {
                String[] splitLine = line.split("\t");
                String ensemblId = splitLine[0];
                String uniprotId = splitLine[1];
                List uniprotIdList = (List)ensemblToUniprotMap.get(ensemblId);
                if (uniprotIdList==null)
                {
                    uniprotIdList = new ArrayList();

                }
                uniprotIdList.add(uniprotId);
                ensemblToUniprotMap.put(ensemblId,uniprotIdList);
            }
            br.close();
        }
        catch(Exception e)
        {
            System.out.println();
        }
    }

}