package org.genemania.dw.util;

import java.sql.SQLException;
import java.util.ArrayList;

import org.genemania.dw.db.EntrezMirrorTables;

/**
 * bioinfo related utils. The current preference is not to have a separate
 * class for species, so its related controlled vocab/functionality is included
 * here. The dumpSpeciesNames () will evolve into the overriden toString ()
 * method in that case.
 *
 * @author rashadbadrawi
 */

public class DWUtil {

    //Controlled vocab for favorite species
    public static final String SP_CRESS = "At";
    public static final String SP_HUMAN = "Hs";
    public static final String SP_MOUSE = "Mm";
    public static final String SP_RAT = "Rn";
    public static final String SP_WORM = "Ce";
    public static final String SP_YEAST = "Sc";
    public static final String SP_FLY = "Dm";
    public static final String SP_ECOLI = "Ec";

    //Entrez species IDs for favorite species
    public static final int TAXID_HS = 9606;
    public static final int TAXID_MM = 10090;
    public static final int TAXID_RN = 10116;
    public static final int TAXID_CE = 6239;
    public static final int TAXID_AT = 3702;
    public static final int TAXID_SC = 4932;
    public static final int TAXID_DM = 7227;
    public static final int TAXID_EC = 562;

    //public static final String SP_NAME_COMMON = "common name";
    public static final String SP_NAME_COMMON = "genbank common name";
    public static final String SP_NAME_SCIENTIFIC = "scientific name";

    private static ArrayList <String> speciesNamesList;

    static {
        speciesNamesList = new ArrayList <String> ();
        speciesNamesList.add (DWUtil.SP_CRESS);
        speciesNamesList.add (DWUtil.SP_HUMAN);
        speciesNamesList.add (DWUtil.SP_MOUSE);
        speciesNamesList.add (DWUtil.SP_RAT);
        speciesNamesList.add (DWUtil.SP_WORM);
        speciesNamesList.add (DWUtil.SP_YEAST);
        speciesNamesList.add (DWUtil.SP_FLY);
        speciesNamesList.add (DWUtil.SP_ECOLI);
    }

    private DWUtil () {}                            //no instances allowed
    
    public static boolean isSupportedSpecies (String speciesName) {

        return speciesNamesList.contains (speciesName);
    }

    public static ArrayList <String> getSpeciesAbbNames () {

        return speciesNamesList;
    }

    public static int getTaxIDByName (String speciesName) {

        if (!DWUtil.isSupportedSpecies(speciesName)) {
            throw new IllegalArgumentException ("Species not supported: " +
                                                speciesName);
        }
        if (DWUtil.SP_CRESS.equals (speciesName)) {
            return DWUtil.TAXID_AT;
        } else if (DWUtil.SP_ECOLI.equals (speciesName)) {
            return DWUtil.TAXID_EC;
        } else if (DWUtil.SP_FLY.equals (speciesName)) {
            return DWUtil.TAXID_DM;
        } else if (DWUtil.SP_HUMAN.equals (speciesName)) {
            return DWUtil.TAXID_HS;
        } else if (DWUtil.SP_MOUSE.equals (speciesName)) {
            return DWUtil.TAXID_MM;
        } else if (DWUtil.SP_RAT.equals (speciesName)) {
            return DWUtil.TAXID_RN;
        } else if (DWUtil.SP_WORM.equals (speciesName)) {
            return DWUtil.TAXID_CE;
        } else {             //if (DWUtil.SP_YEAST.equals (speciesName)) {
            return DWUtil.TAXID_SC;
        }
    }

    public static ArrayList <String> getSpeciesScientificName (int taxID)
                                     throws SQLException {

        ArrayList <String> tempList = EntrezMirrorTables.getSpeciesScientificName(taxID);
        if (tempList.size () > 1) {
            System.err.println ("Warning: More than one scientific name listed: " + taxID);
        }

        return tempList;
    }

    public static ArrayList <String> getSpeciesCommonName (int taxID)
                                     throws SQLException {

        return EntrezMirrorTables.getSpeciesCommonName(taxID);
    }

    public static String dumpSpeciesNames (int taxID) throws SQLException {

        String scNameStr = GenUtil.NA, cmNameStr = "";

        ArrayList <String> scNameList = DWUtil.getSpeciesScientificName(taxID);
        ArrayList <String> cmNameList = DWUtil.getSpeciesCommonName(taxID);
        if (scNameList.size () > 0) {
            scNameStr = scNameList.get (0);
        }
        for (int j = 0; j < cmNameList.size (); j++) {
            cmNameStr += cmNameList.get (j);
            if (j < cmNameList.size () - 1) {
                cmNameStr += GenUtil.SEMICOLON;
            }
        }
        if (cmNameStr.length () == 0) {
            cmNameStr = GenUtil.NA;
        }

        return taxID + GenUtil.TAB + scNameStr + GenUtil.TAB + cmNameStr;
    }
}
