/*
 * Decompiled with CFR 0.152.
 */
package jannovar;

import jannovar.annotation.Annotation;
import jannovar.annotation.AnnotationList;
import jannovar.common.ChromosomeMap;
import jannovar.common.Constants;
import jannovar.exception.AnnotationException;
import jannovar.exception.FileDownloadException;
import jannovar.exception.InvalidAttributException;
import jannovar.exception.JannovarException;
import jannovar.exception.VCFParseException;
import jannovar.exome.Variant;
import jannovar.io.EnsemblFastaParser;
import jannovar.io.GFFparser;
import jannovar.io.RefSeqFastaParser;
import jannovar.io.SerializationManager;
import jannovar.io.TranscriptDataDownloader;
import jannovar.io.UCSCKGParser;
import jannovar.io.VCFLine;
import jannovar.io.VCFReader;
import jannovar.reference.Chromosome;
import jannovar.reference.TranscriptModel;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

public class Jannovar {
    private String dirPath = null;
    private boolean createUCSC;
    private boolean createRefseq;
    private boolean createEnsembl;
    private ArrayList<TranscriptModel> transcriptModelList = null;
    private HashMap<Byte, Chromosome> chromosomeMap = null;
    private final ArrayList<Variant> variantList = null;
    private static final String UCSCserializationFileName = "ucsc_%s.ser";
    private static final String EnsemblSerializationFileName = "ensembl_%s.ser";
    private static final String RefseqSerializationFileName = "refseq_%s.ser";
    private boolean performSerialization = false;
    private String serializedFile = null;
    private String VCFfilePath = null;
    private String proxy = null;
    private String proxyPort = null;
    private boolean jannovarFormat;
    private boolean showAll;
    private Constants.Release genomeRelease = Constants.Release.HG19;
    private String outVCFfolder = null;
    private String chromosomalChange;

    public static void main(String[] argv) {
        Jannovar anno = new Jannovar(argv);
        try {
            if (anno.createUCSC()) {
                anno.downloadTranscriptFiles(0, anno.genomeRelease);
                anno.inputTranscriptModelDataFromUCSCFiles();
                anno.serializeUCSCdata();
                return;
            }
            if (anno.createEnsembl()) {
                anno.downloadTranscriptFiles(1, anno.genomeRelease);
                anno.inputTranscriptModelDataFromEnsembl();
                anno.serializeEnsemblData();
                return;
            }
            if (anno.createRefseq()) {
                anno.downloadTranscriptFiles(2, anno.genomeRelease);
                anno.inputTranscriptModelDataFromRefseq();
                anno.serializeRefseqData();
                return;
            }
        }
        catch (JannovarException e) {
            System.err.println("[ERROR] Error while attempting to download transcript definition files.");
            System.err.println("[ERROR] " + e.toString());
            System.err.println("[ERROR] A common error is the failure to set the network proxy (see tutorial).");
            System.exit(1);
        }
        if (anno.deserialize()) {
            try {
                anno.deserializeTranscriptDefinitionFile();
            }
            catch (JannovarException je) {
                System.out.println("[ERROR] Could not deserialize UCSC data: " + je.toString());
                System.exit(1);
            }
        } else {
            System.err.println("[ERROR] You need to pass ucscs.ser file to perform analysis.");
            Jannovar.usage();
            System.exit(1);
        }
        if (anno.hasVCFfile()) {
            try {
                anno.annotateVCF();
            }
            catch (JannovarException je) {
                System.out.println("[ERROR] Could not annotate VCF data: " + je.toString());
                System.exit(1);
            }
        } else if (anno.chromosomalChange == null) {
            System.out.println("[ERROR] No VCF file found and no chromosomal position and variation was found");
        } else {
            try {
                anno.annotatePosition();
            }
            catch (JannovarException je) {
                System.out.println("[ERROR] Could not annotate input data: " + anno.chromosomalChange);
                System.exit(1);
            }
        }
    }

    public Jannovar(String[] argv) {
        this.parseCommandLineArguments(argv);
        if (!this.dirPath.endsWith(System.getProperty("file.separator"))) {
            this.dirPath = this.dirPath + System.getProperty("file.separator");
        }
        if (this.outVCFfolder != null && !this.outVCFfolder.endsWith(System.getProperty("file.separator"))) {
            this.outVCFfolder = this.outVCFfolder + System.getProperty("file.separator");
        }
    }

    public boolean createUCSC() {
        return this.createUCSC;
    }

    public boolean createRefseq() {
        return this.createRefseq;
    }

    public boolean createEnsembl() {
        return this.createEnsembl;
    }

    public void downloadTranscriptFiles(int source, Constants.Release rel) {
        try {
            TranscriptDataDownloader downloader = this.proxy != null && this.proxyPort != null ? new TranscriptDataDownloader(this.dirPath + this.genomeRelease.getUCSCString(this.genomeRelease), this.proxy, this.proxyPort) : new TranscriptDataDownloader(this.dirPath + this.genomeRelease.getUCSCString(this.genomeRelease));
            downloader.downloadTranscriptFiles(source, rel);
        }
        catch (FileDownloadException e) {
            System.err.println(e);
            System.exit(1);
        }
    }

    public boolean serialize() {
        return this.performSerialization;
    }

    public boolean deserialize() {
        return this.serializedFile != null;
    }

    public boolean hasVCFfile() {
        return this.VCFfilePath != null;
    }

    private void annotateVCFLine(VCFLine line, Variant v, Writer out) throws IOException, AnnotationException {
        byte chr = v.getChromosomeAsByte();
        int pos = v.get_position();
        String ref = v.get_ref();
        String alt = v.get_alt();
        if (alt.charAt(0) == '[' || alt.charAt(0) == ']' || alt.equals(".")) {
            String[] a = line.getOriginalVCFLine().split("\t");
            Chromosome c = this.chromosomeMap.get(chr);
            if (a.length > 1 && c != null) {
                out.write(c.getChromosomeName());
                for (int i = 1; i < a.length; ++i) {
                    out.write("\t" + a[i]);
                }
                out.write("\n");
            } else {
                out.write(line.getOriginalVCFLine() + "\n");
            }
        } else {
            String effect;
            String annotation;
            Chromosome c = this.chromosomeMap.get(chr);
            if (c == null) {
                String e = String.format("Could not identify chromosome \"%d\"", chr);
                throw new AnnotationException(e);
            }
            AnnotationList anno = c.getAnnotationList(pos, ref, alt);
            if (anno == null) {
                String e = String.format("No annotations found for variant %s", v.toString());
                throw new AnnotationException(e);
            }
            if (this.showAll) {
                annotation = anno.getAllTranscriptAnnotations();
                effect = anno.getAllTranscriptVariantEffects();
            } else {
                annotation = anno.getSingleTranscriptAnnotation();
                effect = anno.getVariantType().toString();
            }
            String[] A = line.getOriginalVCFLine().split("\t");
            out.write(c.getChromosomeName() + "\t");
            for (int i = 1; i < 7; ++i) {
                out.write(A[i] + "\t");
            }
            String INFO = A[7].length() > 0 ? String.format("EFFECT=%s;HGVS=%s;%s", effect, annotation, A[7]) : String.format("EFFECT=%s;HGVS=%s", effect, annotation, A[7]);
            out.write(INFO + "\t");
            for (int i = 8; i < A.length; ++i) {
                out.write(A[i] + "\t");
            }
            out.write("\n");
        }
    }

    private void outputJannovarLine(int n, Variant v, Writer out) throws IOException, AnnotationException {
        byte chr = v.getChromosomeAsByte();
        String chrStr = v.get_chromosome_as_string();
        int pos = v.get_position();
        String ref = v.get_ref();
        String alt = v.get_alt();
        String gtype = v.getGenotypeAsString();
        float qual = v.getVariantPhredScore();
        Chromosome c = this.chromosomeMap.get(chr);
        if (c == null) {
            String e = String.format("Could not identify chromosome \"%d\"", chr);
            throw new AnnotationException(e);
        }
        AnnotationList anno = c.getAnnotationList(pos, ref, alt);
        if (anno == null) {
            String e = String.format("No annotations found for variant %s", v.toString());
            throw new AnnotationException(e);
        }
        ArrayList<Annotation> lst = anno.getAnnotationList();
        for (Annotation a : lst) {
            String effect = a.getVariantTypeAsString();
            String annt = a.getVariantAnnotation();
            String sym = a.getGeneSymbol();
            String s = String.format("%d\t%s\t%s\t%s\t%s\t%d\t%s\t%s\t%s\t%.1f", n, effect, sym, annt, chrStr, pos, ref, alt, gtype, Float.valueOf(qual));
            out.write(s + "\n");
        }
    }

    private void outputAnnotatedVCF(VCFReader parser) throws JannovarException {
        int i;
        File f = new File(this.VCFfilePath);
        String outname = f.getName();
        if (this.outVCFfolder != null) {
            outname = this.outVCFfolder + outname;
        }
        if ((i = outname.lastIndexOf("vcf")) < 0) {
            i = outname.lastIndexOf("VCF");
        }
        outname = i < 0 ? outname + ".jv.vcf" : outname.substring(0, i) + "jv.vcf";
        try {
            FileWriter fstream = new FileWriter(outname);
            BufferedWriter out = new BufferedWriter(fstream);
            ArrayList<String> lst = parser.getAnnotatedVCFHeader();
            for (String s : lst) {
                out.write(s + "\n");
            }
            Iterator<VCFLine> iter = parser.getVCFLineIterator();
            while (iter.hasNext()) {
                VCFLine line = iter.next();
                Variant v = line.toVariant();
                try {
                    this.annotateVCFLine(line, v, out);
                }
                catch (AnnotationException e) {
                    System.out.println("[WARN] Annotation error: " + e.toString());
                }
            }
            out.close();
        }
        catch (IOException e) {
            System.out.println("[ERROR] Error writing annotated VCF file");
            System.out.println("[ERROR] " + e.toString());
            System.exit(1);
        }
        System.out.println("[INFO] Wrote annotated VCF file to \"" + outname + "\"");
    }

    private void outputJannovarFormatFile(VCFReader parser) throws JannovarException {
        File f = new File(this.VCFfilePath);
        String outname = f.getName() + ".jannovar";
        try {
            FileWriter fstream = new FileWriter(outname);
            BufferedWriter out = new BufferedWriter(fstream);
            int n = 0;
            Iterator<Variant> iter = parser.getVariantIterator();
            while (iter.hasNext()) {
                ++n;
                Variant v = iter.next();
                try {
                    this.outputJannovarLine(n, v, out);
                }
                catch (AnnotationException e) {
                    System.out.println("[WARN] Annotation error: " + e.toString());
                }
            }
            out.close();
        }
        catch (IOException e) {
            System.err.println("[ERROR] Error writing annotated VCF file");
            System.err.println("[ERROR] " + e.toString());
            System.exit(1);
        }
        System.out.println("[INFO] Wrote annotations to \"" + outname + "\"");
    }

    private void annotatePosition() throws AnnotationException {
        String effect;
        String annotation;
        System.out.println("input: " + this.chromosomalChange);
        Pattern pat = Pattern.compile("(chr[0-9MXY]+):([0-9]+)([ACGTN])>([ACGTN])");
        Matcher mat = pat.matcher(this.chromosomalChange);
        if (!mat.matches() | mat.groupCount() != 4) {
            System.err.println("[ERROR] Input string for the chromosomal change does not fit the regular expression ... :(");
            System.exit(3);
        }
        byte chr = ChromosomeMap.identifier2chromosom.get(mat.group(1));
        int pos = Integer.parseInt(mat.group(2));
        String ref = mat.group(3);
        String alt = mat.group(4);
        Chromosome c = this.chromosomeMap.get(chr);
        if (c == null) {
            String e = String.format("Could not identify chromosome \"%d\"", chr);
            throw new AnnotationException(e);
        }
        AnnotationList anno = c.getAnnotationList(pos, ref, alt);
        if (anno == null) {
            String e = String.format("No annotations found for variant %s", this.chromosomalChange);
            throw new AnnotationException(e);
        }
        if (this.showAll) {
            annotation = anno.getAllTranscriptAnnotations();
            effect = anno.getAllTranscriptVariantEffects();
        } else {
            annotation = anno.getSingleTranscriptAnnotation();
            effect = anno.getVariantType().toString();
        }
        System.out.println(String.format("EFFECT=%s;HGVS=%s", effect, annotation));
    }

    public void annotateVCF() throws JannovarException {
        VCFReader parser = new VCFReader(this.VCFfilePath);
        VCFLine.setStoreVCFLines();
        try {
            parser.inputVCFheader();
        }
        catch (VCFParseException e) {
            System.err.println("[ERROR] Unable to parse VCF file");
            System.err.println(e.toString());
            System.exit(1);
        }
        if (this.jannovarFormat) {
            this.outputJannovarFormatFile(parser);
        } else {
            this.outputAnnotatedVCF(parser);
        }
    }

    public void serializeRefseqData() throws JannovarException {
        SerializationManager manager = new SerializationManager();
        System.out.println("[INFO] Serializing RefSeq data as " + String.format(this.dirPath + RefseqSerializationFileName, this.genomeRelease.getUCSCString(this.genomeRelease)));
        manager.serializeKnownGeneList(String.format(this.dirPath + RefseqSerializationFileName, this.genomeRelease.getUCSCString(this.genomeRelease)), this.transcriptModelList);
    }

    public void serializeEnsemblData() throws JannovarException {
        SerializationManager manager = new SerializationManager();
        System.out.println("[INFO] Serializing Ensembl data as " + String.format(this.dirPath + EnsemblSerializationFileName, this.genomeRelease.getUCSCString(this.genomeRelease)));
        manager.serializeKnownGeneList(String.format(this.dirPath + EnsemblSerializationFileName, this.genomeRelease.getUCSCString(this.genomeRelease)), this.transcriptModelList);
    }

    public void serializeUCSCdata() throws JannovarException {
        SerializationManager manager = new SerializationManager();
        System.out.println("[INFO] Serializing UCSC data as " + String.format(this.dirPath + UCSCserializationFileName, this.genomeRelease.getUCSCString(this.genomeRelease)));
        manager.serializeKnownGeneList(String.format(this.dirPath + UCSCserializationFileName, this.genomeRelease.getUCSCString(this.genomeRelease)), this.transcriptModelList);
    }

    public void deserializeTranscriptDefinitionFile() throws JannovarException {
        SerializationManager manager = new SerializationManager();
        ArrayList<TranscriptModel> kgList = manager.deserializeKnownGeneList(this.serializedFile);
        this.chromosomeMap = Chromosome.constructChromosomeMapWithIntervalTree(kgList);
    }

    private void inputTranscriptModelDataFromRefseq() {
        GFFparser gff = new GFFparser();
        String path = this.dirPath + this.genomeRelease.getUCSCString(this.genomeRelease);
        if (!path.endsWith(System.getProperty("file.separator"))) {
            path = path + System.getProperty("file.separator");
        }
        switch (this.genomeRelease) {
            case MM9: {
                gff.parse(path + "ref_MGSCv37_top_level.gff3.gz");
                break;
            }
            case MM10: {
                gff.parse(path + "ref_GRCm38.p2_top_level.gff3.gz");
                break;
            }
            case HG18: {
                gff.parse(path + "ref_NCBI36_top_level.gff3.gz");
                break;
            }
            case HG19: {
                gff.parse(path + "ref_GRCh37.p13_top_level.gff3.gz");
                break;
            }
            case HG38: {
                gff.parse(path + "ref_GRCh38_top_level.gff3.gz");
                break;
            }
            default: {
                System.err.println("[ERROR] Unknown release: " + (Object)((Object)this.genomeRelease));
                System.exit(20);
            }
        }
        try {
            this.transcriptModelList = gff.getTranscriptModelBuilder().buildTranscriptModels();
        }
        catch (InvalidAttributException e) {
            System.err.println("[ERROR] Unable to input data from the Refseq files");
            e.printStackTrace();
            System.exit(1);
        }
        RefSeqFastaParser efp = new RefSeqFastaParser(path + "rna.fa.gz", this.transcriptModelList);
        int before = this.transcriptModelList.size();
        this.transcriptModelList = efp.parse();
        int after = this.transcriptModelList.size();
        System.out.println(String.format("[INFO] Found %d transcript models from Refseq GFF resource, %d of which had sequences", before, after));
    }

    private void inputTranscriptModelDataFromEnsembl() {
        GFFparser gff = new GFFparser();
        String path = this.dirPath + this.genomeRelease.getUCSCString(this.genomeRelease);
        if (!path.endsWith(System.getProperty("file.separator"))) {
            path = path + System.getProperty("file.separator");
        }
        switch (this.genomeRelease) {
            case MM9: {
                path = path + "Mus_musculus.NCBIM37.67";
                break;
            }
            case MM10: {
                path = path + "Mus_musculus.GRCm38.74";
                break;
            }
            case HG18: {
                path = path + "Homo_sapiens.NCBI36.54";
                break;
            }
            case HG19: {
                path = path + "Homo_sapiens.GRCh37.74";
                break;
            }
            default: {
                System.err.println("[ERROR] Unknown release: " + (Object)((Object)this.genomeRelease));
                System.exit(20);
            }
        }
        gff.parse(path + ".gtf.gz");
        try {
            this.transcriptModelList = gff.getTranscriptModelBuilder().buildTranscriptModels();
        }
        catch (InvalidAttributException e) {
            System.err.println("[ERROR] Unable to input data from the Ensembl files");
            e.printStackTrace();
            System.exit(1);
        }
        EnsemblFastaParser efp = new EnsemblFastaParser(path + ".cdna.all.fa.gz", this.transcriptModelList);
        int before = this.transcriptModelList.size();
        this.transcriptModelList = efp.parse();
        int after = this.transcriptModelList.size();
        System.out.println(String.format("[INFO] Found %d transcript models from Ensembl GFF resource, %d of which had sequences", before, after));
    }

    private void inputTranscriptModelDataFromUCSCFiles() {
        String path = this.dirPath + this.genomeRelease.getUCSCString(this.genomeRelease);
        if (!path.endsWith(System.getProperty("file.separator"))) {
            path = path + System.getProperty("file.separator");
        }
        UCSCKGParser parser = new UCSCKGParser(path);
        try {
            parser.parseUCSCFiles();
        }
        catch (Exception e) {
            System.err.println("[ERROR] Unable to input data from the UCSC files");
            e.printStackTrace();
            System.exit(1);
        }
        this.transcriptModelList = parser.getKnownGeneList();
    }

    public void debugShowChromosomeMap() {
        for (Byte c : this.chromosomeMap.keySet()) {
            Chromosome chromo = this.chromosomeMap.get(c);
            System.out.println("Chrom. " + c + ": " + chromo.getNumberOfGenes() + " genes");
        }
    }

    private void parseCommandLineArguments(String[] args) {
        try {
            Options options = new Options();
            options.addOption(new Option("h", "help", false, "Shows this help"));
            options.addOption(new Option("U", "downloaded-data", true, "Path to directory with previously downloaded transcript definition files."));
            options.addOption(new Option("S", "serialize", false, "Serialize"));
            options.addOption(new Option("D", "deserialize", true, "Path to serialized file with UCSC data"));
            options.addOption(new Option("d", "data", true, "Path to write data storage folder (genome files, serialized files, ...)"));
            options.addOption(new Option("O", "output", true, "Path to output folder for the annotated VCF file"));
            options.addOption(new Option("V", "vcf", true, "Path to VCF file"));
            options.addOption(new Option("a", "showall", false, "report annotations for all transcripts to VCF file"));
            options.addOption(new Option("P", "position", true, "chromosomal position to HGVS (e.g. chr1:909238G>C)"));
            options.addOption(new Option("J", "janno", false, "Output Jannovar format"));
            options.addOption(new Option("g", "genome", true, "genome build (mm9, mm10, hg18, hg19, hg38 - only refseq), default hg19"));
            options.addOption(new Option(null, "create-ucsc", false, "Create UCSC definition file"));
            options.addOption(new Option(null, "create-refseq", false, "Create RefSeq definition file"));
            options.addOption(new Option(null, "create-ensembl", false, "Create Ensembl definition file"));
            options.addOption(new Option(null, "proxy", true, "FTP Proxy"));
            options.addOption(new Option(null, "proxy-port", true, "FTP Proxy Port"));
            GnuParser parser = new GnuParser();
            CommandLine cmd = parser.parse(options, args);
            if (cmd.hasOption("h") || cmd.hasOption("H") || args.length == 0) {
                HelpFormatter formatter = new HelpFormatter();
                formatter.printHelp("java -jar Jannovar.jar [-options]", options);
                Jannovar.usage();
                System.exit(0);
            }
            this.jannovarFormat = cmd.hasOption("J");
            this.showAll = cmd.hasOption('a');
            if (cmd.hasOption("create-ucsc")) {
                this.createUCSC = true;
                this.performSerialization = true;
            } else {
                this.createUCSC = false;
            }
            if (cmd.hasOption("create-refseq")) {
                this.createRefseq = true;
                this.performSerialization = true;
            } else {
                this.createRefseq = false;
            }
            if (cmd.hasOption("create-ensembl")) {
                this.createEnsembl = true;
                this.performSerialization = true;
            } else {
                this.createEnsembl = false;
            }
            this.dirPath = cmd.hasOption('d') ? cmd.getOptionValue('d') : "data/";
            if (!this.dirPath.endsWith(System.getProperty("file.separator"))) {
                this.dirPath = this.dirPath + System.getProperty("file.separator");
            }
            if (cmd.hasOption("genome")) {
                String g = cmd.getOptionValue("genome");
                if (g.equals("mm9")) {
                    this.genomeRelease = Constants.Release.MM9;
                }
                if (g.equals("mm10")) {
                    this.genomeRelease = Constants.Release.MM10;
                }
                if (g.equals("hg18")) {
                    this.genomeRelease = Constants.Release.HG18;
                }
                if (g.equals("hg19")) {
                    this.genomeRelease = Constants.Release.HG19;
                }
                if (g.equals("hg38")) {
                    if (this.createRefseq) {
                        this.genomeRelease = Constants.Release.HG38;
                    } else {
                        System.out.println("[INFO] Genome release hg38 only available for Refseq");
                        System.exit(0);
                    }
                }
            } else {
                if (this.performSerialization) {
                    System.out.println("[INFO] Genome release set to default: hg19");
                }
                this.genomeRelease = Constants.Release.HG19;
            }
            if (cmd.hasOption('O')) {
                this.outVCFfolder = cmd.getOptionValue('O');
                File file = new File(this.outVCFfolder);
                if (!file.exists()) {
                    file.mkdirs();
                }
            }
            if (cmd.hasOption('S')) {
                this.performSerialization = true;
            }
            if (cmd.hasOption("proxy")) {
                this.proxy = cmd.getOptionValue("proxy");
            }
            if (cmd.hasOption("proxy-port")) {
                this.proxyPort = cmd.getOptionValue("proxy-port");
            }
            if (cmd.hasOption("U")) {
                this.dirPath = Jannovar.getRequiredOptionValue(cmd, 'U');
            }
            if (cmd.hasOption('D')) {
                this.serializedFile = cmd.getOptionValue('D');
            }
            if (cmd.hasOption("V")) {
                this.VCFfilePath = cmd.getOptionValue("V");
            }
            if (cmd.hasOption("P")) {
                this.chromosomalChange = cmd.getOptionValue("P");
            }
        }
        catch (ParseException pe) {
            System.err.println("Error parsing command line options");
            System.err.println(pe.getMessage());
            System.exit(1);
        }
    }

    private static String getRequiredOptionValue(CommandLine cmd, char name) {
        String val = cmd.getOptionValue(name);
        if (val == null) {
            System.err.println("Aborting because the required argument \"-" + name + "\" wasn't specified! Use the -h for more help.");
            System.exit(-1);
        }
        return val;
    }

    private static void usage() {
        System.out.println("***   Jannovar: Usage     ****");
        System.out.println("Use case 1: Download UCSC data and create transcript data file (ucsc_hg19.ser)");
        System.out.println("$ java -jar Jannovar.jar --create-ucsc [-U name of output directory]");
        System.out.println("Use case 2: Add annotations to a VCF file");
        System.out.println("$ java -jar Jannovar.jar -D ucsc_hg19.ser -V example.vcf");
        System.out.println("Use case 3: Write new file with Jannovar-format annotations of a VCF file");
        System.out.println("$ java -jar Jannovar -D ucsc_hg19.ser -V vcfPath -J");
        System.out.println("*** See the tutorial for details ***");
    }
}

