/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import java.io.Serializable;

import mt.DenseMatrix;

/**
 * Singular value decomposition. To compute a decomposition, use the
 * {@link mt.fact.SingularvalueComputer SingularvalueComputer}.
 */
public class SVD implements Serializable {

	private static final long serialVersionUID = -7712448982013060966L;

	/**
	 * The singular values
	 */
	private double[] S;

	/**
	 * Singular vectors
	 */
	private DenseMatrix U, Vt;

	/**
	 * Allocates storage for an SVD
	 * 
	 * @param m
	 *            Number of rows
	 * @param n
	 *            Number of columns
	 * @param vectors
	 *            True to store the singular vectors
	 */
	public SVD(int m, int n, boolean vectors) {
		S = new double[Math.min(m, n)];

		if (vectors) {
			U = new DenseMatrix(m, m);
			Vt = new DenseMatrix(n, n);
		}
	}

	/**
	 * Allocates storage for an SVD. Stores singular values as well as
	 * singular vectors
	 * 
	 * @param m
	 *            Number of rows
	 * @param n
	 *            Number of columns
	 */
	public SVD(int m, int n) {
		this(m, n, true);
	}

	/**
	 * True if singular vectors are stored
	 */
	public boolean hasSingularVectors() {
		return U != null;
	}

	/**
	 * Returns the left singular vectors, column-wise. Not available for
	 * partial decompositions
	 */
	public DenseMatrix getU() {
		return U;
	}

	/**
	 * Returns the right singular vectors, row-wise. Not available for
	 * partial decompositions
	 */
	public DenseMatrix getVt() {
		return Vt;
	}

	/**
	 * Returns the singular values
	 */
	public double[] getS() {
		return S;
	}

}
