/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt;

import mt.DenseVector;
import mt.Matrix;
import mt.Vector;

/**
 * Matrix stored using an array of <code>HashVector</code>, row-wise
 */
public class HashRowMatrix extends AbstractRowMatrix {

	/**
	 * Constructor for HashRowMatrix
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of column
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 */
	/*public HashRowMatrix(int numRows, int numColumns, int numThreads) {
		super(numRows, numColumns, new HashVector(numColumns), numThreads);
	}*/

	/**
	 * Constructor for HashRowMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 * @param deep
	 *            True for a deep copy, false for a reference copy. A reference
	 *            copy can only be made of a <code>HashRowMatrix</code>
	 */
	/*public HashRowMatrix(Matrix A, int numThreads, boolean deep) {
		super(A, new HashVector(A.numColumns()), numThreads, deep);
	}*/

	/**
	 * Constructor for HashRowMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from. The copy will be deep
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 */
	/*public HashRowMatrix(Matrix A, int numThreads) {
		super(A, new HashVector(A.numColumns()), numThreads);
	}*/

	/**
	 * Constructor for HashRowMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of column
	 */
	public HashRowMatrix(int numRows, int numColumns) {
		super(numRows, numColumns, new HashVector(numColumns));
	}

	/**
	 * Constructor for HashRowMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param A
	 *            Matrix to copy contents from. The copy will be deep
	 */
	public HashRowMatrix(Matrix A) {
		super(A, new HashVector(A.numColumns()));
	}

	/**
	 * Constructor for HashRowMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param A
	 *            Matrix to copy contents from
	 * @param deep
	 *            True for a deep copy, false for a reference copy. A reference
	 *            copy can only be made of a <code>HashRowMatrix</code>
	 */
	public HashRowMatrix(Matrix A, boolean deep) {
		super(A, new HashVector(A.numColumns()), deep);
	}

	public Vector transMultAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.transMultAdd(alpha, x, beta, y, z);

		checkTransMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		// z = beta/alpha * y
		z.set(beta / alpha, y);

		// z = A'x + z
		for (int i = 0; i < numRows; ++i) {
			HashVector v = (HashVector) rowD[i];
			int[] index = v.getIndices();
			double[] data = v.getValues();
			boolean[] used = v.getUsed();
			for (int j = 0; j < used.length; ++j)
				if (used[j])
					zd[index[j]] += data[j] * xd[i];
		}

		// z = alpha*z = alpha * A'x + beta*y
		return z.scale(alpha);
	}

	//public HashRowMatrix copy() {
	public Matrix copy() {
		return new HashRowMatrix(this);
	}

	//public HashVector getRow(int i) {
	public Vector getRow(int i) {
		return (HashVector) rowD[i];
	}

	/**
	 * Sets the given row equal the passed vector
	 */
	public void setRow(int i, HashVector x) {
		if (x.size() != numColumns)
			throw new IllegalArgumentException("New row must be of the same size as existing row");
		rowD[i] = x;
	}

}
