/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import java.util.Arrays;
import java.util.Iterator;

//import mt.ll.BLASkernel;
import mt.ll.BLASkernel.Transpose;
import mt.ll.Interface;

/**
 * Dense matrix. It is a good all-round matrix structure, with fast access and
 * efficient algebraic operations. The matrix
 * <p>
 * <table border="1">
 * <tr>
 * <td>a<sub>11</sub></td>
 * <td>a<sub>12</sub></td>
 * <td>a<sub>13</sub></td>
 * <td>a<sub>14</sub></td>
 * </tr>
 * <tr>
 * <td>a<sub>21</sub></td>
 * <td>a<sub>22</sub></td>
 * <td>a<sub>23</sub></td>
 * <td>a<sub>24</sub></td>
 * </tr>
 * <tr>
 * <td>a<sub>31</sub></td>
 * <td>a<sub>32</sub></td>
 * <td>a<sub>33</sub></td>
 * <td>a<sub>34</sub></td>
 * </tr>
 * <tr>
 * <td>a<sub>41</sub></td>
 * <td>a<sub>42</sub></td>
 * <td>a<sub>43</sub></td>
 * <td>a<sub>44</sub></td>
 * </tr>
 * </table>
 * </p>
 * <p>
 * is stored column major in a single array, as follows:
 * </p>
 * <p>
 * <table border="1">
 * <tr>
 * <td>a<sub>11</sub></td>
 * <td>a<sub>21</sub></td>
 * <td>a<sub>31</sub></td>
 * <td>a<sub>41</sub></td>
 * <td>a<sub>12</sub></td>
 * <td>a<sub>22</sub></td>
 * <td>a<sub>32</sub></td>
 * <td>a<sub>42</sub></td>
 * <td>a<sub>13</sub></td>
 * <td>a<sub>23</sub></td>
 * <td>a<sub>33</sub></td>
 * <td>a<sub>43</sub></td>
 * <td>a<sub>14</sub></td>
 * <td>a<sub>24</sub></td>
 * <td>a<sub>34</sub></td>
 * <td>a<sub>44</sub></td>
 * </tr>
 * </table>
 * </p>
 */
public class DenseMatrix extends AbstractDenseMatrix {

	private static final long serialVersionUID = -7925057485928464167L;

	/**
	 * Constructor for DenseMatrix
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of columns
	 */
	public DenseMatrix(int numRows, int numColumns) {
		super(numRows, numColumns);
	}

	/**
	 * Constructor for DenseMatrix
	 * 
	 * @param A
	 *            Matrix to copy. A deep copy is made
	 */
	public DenseMatrix(Matrix A) {
		super(A);
	}

	/**
	 * Constructor for DenseMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from
	 * @param deep
	 *            If true, <code>A</code> is copied, else a shallow copy is
	 *            made and the matrices share underlying storage. For this,
	 *            <code>A</code> must be a dense matrix
	 */
	public DenseMatrix(Matrix A, boolean deep) {
		super(A, deep);
	}

	/**
	 * Constructor for DenseMatrix. Builds the matrix from a vector
	 * 
	 * @param x
	 *            Vector to copy from. This will form this matrix' single
	 *            column
	 * @param deep
	 *            If true, x is copied, if false, the internal storage of this
	 *            matrix is the same as that of the vector. In that case,
	 *            <code>x</code> must be a <code>DenseVector</code>
	 */
	public DenseMatrix(Vector x, boolean deep) {
		super(x.size(), 1);

		/*if (deep)
			for (VectorEntry e : x)
				set(e.index(), 0, e.get());*/
		if(deep) {
			VectorEntry e;
			Iterator iter = x.iterator();
			while(iter.hasNext()) {
				e = (VectorEntry) iter.next();
				set(e.index(), 0, e.get());
			}
		}
		else {
			if (!(x instanceof DenseVector))
				throw new IllegalArgumentException("x must be a DenseVector");
			data = ((DenseVector) x).getData();
		}
	}

	/**
	 * Constructor for DenseMatrix. Builds the matrix from a vector
	 * 
	 * @param x
	 *            The vector which forms this matrix' single column. It is
	 *            copied, not referenced
	 */
	public DenseMatrix(Vector x) {
		this(x, true);
	}

	/**
	 * Constructor for DenseMatrix. Builds the matrix from vectors. Each vector
	 * will correspond to a column of the matrix
	 * 
	 * @param x
	 *            Vectors which forms the columns of this matrix. Every vector
	 *            must have the same size
	 */
	public DenseMatrix(Vector[] x) {
		super(x[0].size(), x.length);

		// Ensure correct sizes
		/*for (Vector v : x)
			if (v.size() != numRows)
				throw new IllegalArgumentException("All vectors must be of the same size");*/
		for(int i = 0; i < x.length; ++i) {
			if (x[i].size() != numRows)
				throw new IllegalArgumentException("All vectors must be of the same size");
		}
		
		// Copy the contents
		for (int j = 0; j < x.length; ++j) {
			VectorEntry e;
			Iterator iter = x[j].iterator();
			while(iter.hasNext()) {
				e = (VectorEntry) iter.next();
				set(e.index(), j, e.get());
			}
		}
			/*for (VectorEntry e : x[j])
				set(e.index(), j, e.get());*/
			
	}

	/**
	 * Constructor for DenseMatrix. Copies from the passed array
	 * 
	 * @param values
	 *            Arrays to copy from. Every sub-array must have the same size
	 */
	public DenseMatrix(double[][] values) {
		super(values.length, values[0].length);

		// Copy the contents
		for (int i = 0; i < values.length; ++i) {
			if (values[i].length != numColumns)
				throw new IllegalArgumentException("Array cannot be jagged");
			for (int j = 0; j < values[i].length; ++j)
				set(i, j, values[i][j]);
		}
	}

	//public DenseMatrix copy() {
	public Matrix copy() {
		return new DenseMatrix(this);
	}

	void copy(Matrix A) {
		/*for (MatrixEntry e : A)
			set(e.row(), e.column(), e.get());*/
		MatrixEntry e;
		Iterator iter = A.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			set(e.row(), e.column(), e.get());
		}
	}

	public Matrix multAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		if (!(B instanceof DenseMatrix) || !(D instanceof DenseMatrix))
			return super.multAdd(alpha, B, beta, C, D);

		checkMultAdd(B, C, D);

		double[] Bd = ((DenseMatrix) B).getData(),
			Dd = ((DenseMatrix) D).getData();

		D.set(C);

		Interface.blas().gemm(
			Transpose.NoTranspose,
			Transpose.NoTranspose,
			D.numRows(),
			C.numColumns(),
			numColumns,
			alpha,
			data,
			Math.max(1, numRows),
			Bd,
			Math.max(1, B.numRows()),
			beta,
			Dd,
			Math.max(1, D.numRows()));

		return D;
	}

	public Matrix transAmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		if (!(B instanceof DenseMatrix) || !(D instanceof DenseMatrix))
			return super.transAmultAdd(alpha, B, beta, C, D);

		checkTransAmultAdd(B, C, D);

		double[] Bd = ((DenseMatrix) B).getData(),
			Dd = ((DenseMatrix) D).getData();

		D.set(C);

		Interface.blas().gemm(
			Transpose.Transpose,
			Transpose.NoTranspose,
			D.numRows(),
			C.numColumns(),
			numRows,
			alpha,
			data,
			Math.max(1, numRows),
			Bd,
			Math.max(1, B.numRows()),
			beta,
			Dd,
			Math.max(1, D.numRows()));

		return D;
	}

	public Matrix transBmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		if (!(B instanceof DenseMatrix) || !(D instanceof DenseMatrix))
			return super.transBmultAdd(alpha, B, beta, C, D);

		checkTransBmultAdd(B, C, D);

		double[] Bd = ((DenseMatrix) B).getData(),
			Dd = ((DenseMatrix) D).getData();

		D.set(C);

		Interface.blas().gemm(
			Transpose.NoTranspose,
			Transpose.Transpose,
			D.numRows(),
			C.numColumns(),
			numColumns,
			alpha,
			data,
			Math.max(1, numRows),
			Bd,
			Math.max(1, B.numRows()),
			beta,
			Dd,
			Math.max(1, D.numRows()));

		return D;
	}

	public Matrix transABmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		if (!(B instanceof DenseMatrix) || !(D instanceof DenseMatrix))
			return super.transABmultAdd(alpha, B, beta, C, D);

		checkTransABmultAdd(B, C, D);

		double[] Bd = ((DenseMatrix) B).getData(),
			Dd = ((DenseMatrix) D).getData();

		D.set(C);

		Interface.blas().gemm(
			Transpose.Transpose,
			Transpose.Transpose,
			D.numRows(),
			C.numColumns(),
			numRows,
			alpha,
			data,
			Math.max(1, numRows),
			Bd,
			Math.max(1, B.numRows()),
			beta,
			Dd,
			Math.max(1, D.numRows()));

		return D;
	}

	public Matrix rank1(double alpha, Vector x, Vector y) {
		if (!(x instanceof DenseVector) || !(y instanceof DenseVector))
			return super.rank1(alpha, x, y);

		checkRank1(x, y);

		double[] xd = ((DenseVector) x).getData(),
			yd = ((DenseVector) y).getData();

		Interface.blas().ger(
			numRows,
			numColumns,
			alpha,
			xd,
			yd,
			data,
			Math.max(1, numRows));

		return this;
	}

	public Vector multAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.multAdd(alpha, x, beta, y, z);

		checkMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		z.set(y);

		Interface.blas().gemv(
			Transpose.NoTranspose,
			numRows,
			numColumns,
			alpha,
			data,
			Math.max(numRows, 1),
			xd,
			beta,
			zd);

		return z;
	}

	public Vector transMultAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.transMultAdd(alpha, x, beta, y, z);

		checkTransMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		z.set(y);

		Interface.blas().gemv(
			Transpose.Transpose,
			numRows,
			numColumns,
			alpha,
			data,
			Math.max(numRows, 1),
			xd,
			beta,
			zd);

		return z;
	}

	public Matrix solve(Matrix B, Matrix X) {
		// We allow non-square matrices, as we then use a least-squares solver
		if (B.numRows() != numRows)
			throw new IllegalArgumentException("B.numRows() != A.numRows()");
		if (B.numColumns() != X.numColumns())
			throw new IllegalArgumentException("B.numColumns() != X.numColumns()");
		if (X.numRows() != numColumns)
			throw new IllegalArgumentException("X.numRows() != A.numColumns()");

		if (isSquare())
			return LUsolve(B, X);
		else
			return QRsolve(B, X, Transpose.NoTranspose);
	}

	public Vector solve(Vector b, Vector x) {
		DenseMatrix B = new DenseMatrix(b, false),
			X = new DenseMatrix(x, false);
		solve(B, X);
		return x;
	}

	public Matrix transSolve(Matrix B, Matrix X) {
		// We allow non-square matrices, as we then use a least-squares solver
		if (B.numRows() != numColumns)
			throw new IllegalArgumentException("B.numRows() != A.numColumns()");
		if (B.numColumns() != X.numColumns())
			throw new IllegalArgumentException("B.numColumns() != X.numColumns()");
		if (X.numRows() != numRows)
			throw new IllegalArgumentException("X.numRows() != A.numRows()");

		return QRsolve(B, X, Transpose.Transpose);
	}

	public Vector transSolve(Vector b, Vector x) {
		DenseMatrix B = new DenseMatrix(b, false),
			X = new DenseMatrix(x, false);
		transSolve(B, X);
		return x;
	}

	Matrix LUsolve(Matrix B, Matrix X) {
		if (!(X instanceof DenseMatrix))
			throw new UnsupportedOperationException("X must be a DenseMatrix");

		double[] Xd = ((DenseMatrix) X).getData();

		X.set(B);

		int[] piv = new int[numRows];

		int info =
			Interface.lapack().gesv(
				numRows,
				B.numColumns(),
				(double[]) data.clone(),
				piv,
				Xd);

		if (info > 0)
			throw new MatrixSingularException();
		else if (info < 0)
			throw new IllegalArgumentException();

		return X;
	}

	Matrix QRsolve(Matrix B, Matrix X, Transpose trans) {
		int nrhs = B.numColumns();

		// Allocate temporary solution matrix
		DenseMatrix Xtmp = new DenseMatrix(Math.max(numRows, numColumns), nrhs);
		int M = trans == Transpose.NoTranspose ? numRows : numColumns;
		for (int j = 0; j < nrhs; ++j)
			for (int i = 0; i < M; ++i)
				Xtmp.set(i, j, B.get(i, j));
		double[] newData = (double[]) data.clone();

		// Query optimal workspace
		double[] work = new double[1];
		int info =
			Interface.lapack().gels(
				trans,
				numRows,
				numColumns,
				nrhs,
				newData,
				Xtmp.getData(),
				work,
				-1);

		// Allocate workspace
		int lwork = -1;
		if (info != 0)
			lwork =
				Math.max(
					1,
					Math.min(numRows, numColumns)
						+ Math.max(Math.min(numRows, numColumns), nrhs));
		else
			lwork = Math.max((int) work[0], 1);
		work = new double[lwork];

		// Compute the factorization
		info =
			Interface.lapack().gels(
				trans,
				numRows,
				numColumns,
				nrhs,
				newData,
				Xtmp.getData(),
				work,
				lwork);

		if (info < 0)
			throw new IllegalArgumentException();

		// Extract the solution
		int N = trans == Transpose.NoTranspose ? numColumns : numRows;
		for (int j = 0; j < nrhs; ++j)
			for (int i = 0; i < N; ++i)
				X.set(i, j, Xtmp.get(i, j));
		return X;
	}

	public Matrix set(Matrix B) {
		if (B == this)
			return this;
		if (!(B instanceof DenseMatrix))
			return super.set(B);

		checkSize(B);

		double[] Bd = ((DenseMatrix) B).getData();

		if (Bd == data)
			return this;

		System.arraycopy(Bd, 0, data, 0, data.length);

		return this;
	}

	public Matrix zero() {
		Arrays.fill(data, 0);
		return this;
	}

}
