/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import java.io.Serializable;

import mt.AbstractMatrix;
import mt.DenseMatrix;
import mt.DenseVector;
import mt.Matrix;
import mt.Vector;
import mt.ll.BLASkernel.Transpose;

/**
 * Orthonormal matrix. Created as part of orthogonal decompositions, and
 * not by the user.
 */
public abstract class OrthMatrix extends AbstractMatrix implements Serializable {

	/**
	 * Scales for the reflectors
	 */
	double[] tau;

	/**
	 * The Householder reflectors
	 */
	DenseMatrix H;

	/**
	 * Constructor for OrthMatrix
	 * 
	 * @param H
	 *            Householder reflectors
	 * @param tau
	 *            Scales for the reflectors
	 */
	OrthMatrix(DenseMatrix H, double[] tau) {
		super(H);
		this.H = H;
		this.tau = tau;
	}

	/**
	 * Generates an explicit representation of the orthogonal matrix
	 */
	public abstract DenseMatrix generate();

	public Matrix multAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		if (!(D instanceof DenseMatrix))
			throw new IllegalArgumentException("D must be a DenseMatrix");

		checkMultAdd(B, C, D);

		// D = alpha*B
		D.set(alpha, B);

		// D = A*D
		mult((DenseMatrix) D, Transpose.NoTranspose);

		// D = D + beta*C
		return D.add(beta, C);
	}

	public Matrix transAmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		if (!(D instanceof DenseMatrix))
			throw new IllegalArgumentException("D must be a DenseMatrix");

		checkTransAmultAdd(B, C, D);

		// D = alpha*B
		D.set(alpha, B);

		// D = A'*D
		mult((DenseMatrix) D, Transpose.Transpose);

		// D = D + beta*C
		return D.add(beta, C);
	}

	/**
	 * Performs an in-place multiplication
	 */
	abstract void mult(DenseMatrix D, Transpose trans);

	public Vector multAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(z instanceof DenseVector))
			throw new IllegalArgumentException("z must be a DenseVector");

		checkMultAdd(x, y, z);

		// z = alpha*x
		z.set(alpha, x);

		// z = A*z
		mult(new DenseMatrix(z, false), Transpose.NoTranspose);

		// z = z + beta*y
		return z.add(beta, y);
	}

	public Vector transMultAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(z instanceof DenseVector))
			throw new IllegalArgumentException("z must be a DenseVector");

		checkTransMultAdd(x, y, z);

		// z = alpha*x
		z.set(alpha, x);

		// z = A'*z
		mult(new DenseMatrix(z, false), Transpose.Transpose);

		// z = z + beta*y
		return z.add(beta, y);
	}

	public Matrix solve(Matrix B, Matrix X) {
		return transAmult(B, X);
	}

	public Vector solve(Vector b, Vector x) {
		return transMult(b, x);
	}

	public Matrix transSolve(Matrix B, Matrix X) {
		return mult(B, X);
	}

	public Vector transSolve(Vector b, Vector x) {
		return mult(b, x);
	}

}
