/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import java.io.Serializable;

import mt.DenseMatrix;
import mt.LowerTriangDenseMatrix;
import mt.UpperTriangDenseMatrix;

/**
 * Orthogonal matrix decomposition
 */
public class OrthogonalDecomposition implements Serializable {

	private static final long serialVersionUID = -1973650562208779301L;

	/**
	 * Scales for the reflectors
	 */
	private double[] tau;

	/**
	 * The Householder reflectors. Also used to extract the triangular factor
	 */
	private DenseMatrix H;

	/**
	 * The orthogonal matrix
	 */
	private OrthMatrix Q;

	/**
	 * Lower triangular factor. May not be present
	 */
	private LowerTriangDenseMatrix L;

	/**
	 * Upper triangular factor. May not be present
	 */
	private UpperTriangDenseMatrix R;

	/**
	 * Allocates storage for a new orthogonal decomposition. 
	 * 
	 * @param m
	 *            Number of rows
	 * @param n
	 *            Number of columns
	 */
	public OrthogonalDecomposition(int m, int n) {
		tau = new double[Math.min(m, n)];
		H = new DenseMatrix(m, n);
	}

	/**
	 * Returns the scales of the Householder reflectors
	 */
	public double[] getTau() {
		return tau;
	}

	/**
	 * Returns the Householder reflectors
	 */
	public DenseMatrix getH() {
		return H;
	}

	/**
	 * Returns the orthogonal part of the factorization
	 */
	public OrthMatrix getQ() {
		return Q;
	}

	/**
	 * Sets the orthogonal matrix
	 */
	public void setQ(OrthMatrix Q) {
		this.Q = Q;
	}

	/**
	 * Returns the lower triangular factor. Should only be called if such a
	 * factor is actually present, else the resulting matrix will be
	 * inconsistent. Note that the returned matrix is a view into the
	 * Householder matrix, therefore modifications should only be done on
	 * copies
	 */
	public LowerTriangDenseMatrix getL() {
		return L;
	}

	/**
	 * Returns the upper triangular factor. Should only be called if such a
	 * factor is actually present, else the resulting matrix will be
	 * inconsistent. Note that the returned matrix is a view into the
	 * Householder matrix, therefore modifications should only be done on
	 * copies
	 */
	public UpperTriangDenseMatrix getR() {
		return R;
	}

	/**
	 * Sets the lower triangular factor
	 */
	public void setL(LowerTriangDenseMatrix L) {
		this.L = L;
	}

	/**
	 * Sets the upper triangular factor
	 */
	public void setR(UpperTriangDenseMatrix R) {
		this.R = R;
	}

}
