/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.Matrix;
import mt.SymmTridiagMatrix;
import mt.ll.Interface;
import mt.ll.LAPACKkernel.JobEig;
import mt.ll.LAPACKkernel.JobEigRange;

/**
 * Computes eigenvalues of symmetrical, tridiagonal matrices
 */
public class SymmTridiagEigenvalueComputer {

	/**
	 * Double work array
	 */
	private double[] work;

	/**
	 * Integer work array
	 */
	private int[] iwork;

	/**
	 * Size of the matrix
	 */
	private int n;

	/**
	 * Job to do
	 */
	private JobEig job;

	/**
	 * Range of eigenvalues to compute
	 */
	private JobEigRange range;

	/**
	 * Eigenvector supports
	 */
	private int[] isuppz;

	/**
	 * Tolerance criteria
	 */
	private double abstol;

	/**
	 * Creates a new eigenvalue computer for symmetrical, tridiagonal matrices.
	 * Computes all eigenvalues and eigenvectors, and uses a default tolerance
	 * criteria
	 * 
	 * @param n
	 *            Size of the matrix
	 */
	public SymmTridiagEigenvalueComputer(int n) {
		this(n, JobEig.All);
	}

	/**
	 * Creates a new eigenvalue computer for symmetrical, tridiagonal matrices.
	 * Computes all eigenvalues and eigenvectors
	 * 
	 * @param n
	 *            Size of the matrix
	 * @param abstol
	 *            Absolute tolerance criteria
	 */
	public SymmTridiagEigenvalueComputer(int n, double abstol) {
		this(n, JobEig.All, abstol);
	}

	/**
	 * Creates a new eigenvalue computer for symmetrical, tridiagonal matrices
	 * 
	 * @param n
	 *            Size of the matrix
	 * @param job
	 *            The type of computation to perform
	 * @param abstol
	 *            Absolute tolerance criteria
	 */
	public SymmTridiagEigenvalueComputer(int n, JobEig job, double abstol) {
		this.n = n;
		this.job = job;
		this.abstol = abstol;

		range = JobEigRange.All;
		isuppz = new int[2 * Math.max(1, n)];

		// Find the needed workspace
		work = new double[1];
		iwork = new int[1];
		int info =
			Interface.lapack().stevr(
				job,
				range,
				n,
				new double[0],
				new double[0],
				0,
				0,
				0,
				0,
				abstol,
				new int[1],
				new double[0],
				new double[0],
				isuppz,
				work,
				-1,
				iwork,
				-1);

		// Allocate workspace
		int lwork = 0, liwork = 0;
		if (info != 0) {
			lwork = 20 * n;
			liwork = 10 * n;
		} else {
			lwork = (int) work[0];
			liwork = iwork[0];
		}
		lwork = Math.max(1, lwork);
		liwork = Math.max(1, liwork);
		work = new double[lwork];
		iwork = new int[liwork];
	}

	/**
	 * Creates a new eigenvalue computer for symmetrical, tridiagonal matrices.
	 * Uses a default tolerance criteria
	 * 
	 * @param n
	 *            Size of the matrix
	 * @param job
	 *            The type of computation to perform
	 */
	public SymmTridiagEigenvalueComputer(int n, JobEig job) {
		this(n, job, Interface.lapack().lamch("Safe minimum"));
	}

	/**
	 * Convience method for computing the full eigenvalue decomposition of the
	 * given matrix
	 * 
	 * @param A
	 *            Matrix to factorize. Main diagonal and superdiagonal is
	 *            copied, and the matrix is not modified
	 * @return
	 *            Newly allocated decomposition
	 * @throws NotConvergedException
	 */
	public static SymmEigenvalueDecomposition factorize(Matrix A)
		throws NotConvergedException {
		return new SymmTridiagEigenvalueComputer(A.numRows()).factor(
			new SymmTridiagMatrix(A));
	}

	/**
	 * Computes the eigenvalue decomposition of the given matrix
	 * 
	 * @param A
	 *            Matrix to factorize. Overwritten on return
	 * @return
	 *            Newly allocated decomposition
	 * @throws NotConvergedException
	 */
	public SymmEigenvalueDecomposition factor(SymmTridiagMatrix A)
		throws NotConvergedException {
		return factor(A, new SymmEigenvalueDecomposition(n, job));
	}

	/**
	 * Computes the eigenvalue decomposition of the given matrix
	 * 
	 * @param A
	 *            Matrix to factorize. Overwritten on return
	 * @param evd
	 *            Factorization is written here
	 * @return
	 *            evd
	 * @throws NotConvergedException
	 */
	public SymmEigenvalueDecomposition factor(
		SymmTridiagMatrix A,
		SymmEigenvalueDecomposition evd)
		throws NotConvergedException {
		if (A.numRows() != n)
			throw new IllegalArgumentException("A.numRows() != n");
		else if (evd.getEigenvalues().length != n)
			throw new IllegalArgumentException("evd.getEigenvalues().length != n");
		else if (job == JobEig.All) {
			if (!evd.hasEigenvectors())
				throw new IllegalArgumentException("Eigenvalue decomposition does not store eigenvectors");
			else if (evd.getEigenvectors().numRows() != n)
				throw new IllegalArgumentException("evd.getEigenvectors().numRows() != n");
			else if (evd.getEigenvectors().numColumns() != n)
				throw new IllegalArgumentException("evd.getEigenvectors().numColumns() != n");
		}

		int info =
			Interface.lapack().stevr(
				job,
				range,
				n,
				A.getDiagonal(),
				A.getOffDiagonal(),
				0,
				0,
				0,
				0,
				abstol,
				new int[1],
				evd.getEigenvalues(),
				job == JobEig.All
					? evd.getEigenvectors().getData()
					: new double[0],
				isuppz,
				work,
				work.length,
				iwork,
				iwork.length);

		if (info > 0)
			throw new NotConvergedException(
				NotConvergedException.Reason.Iterations);
		else if (info < 0)
			throw new IllegalArgumentException();

		return evd;
	}

}
