/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.eig;

import mt.fact.NotConvergedException;

/**
 * Monitor which declares convergence when the residual is below a certain
 * absolute threshold
 */
public class DefaultEigenvalueIterationMonitor
	extends AbstractEigenvalueIterationMonitor {

	/**
	 * Convergence and divergence tolerances
	 */
	private double ctol, dtol;

	/**
	 * Max number of iterations
	 */
	private int maxIter;

	/**
	 * Constructor for DefaultEigenvalueIterationMonitor.
	 * 
	 * @param maxIter
	 *            Maximum number of iterations
	 * @param ctol
	 *            How small the residual must be before convergence is declared
	 * @param dtol
	 *            How large the residual can be before divergence is declared
	 */
	public DefaultEigenvalueIterationMonitor(
		int maxIter,
		double ctol,
		double dtol) {
		this.maxIter = maxIter;
		this.ctol = ctol;
		this.dtol = dtol;
	}

	/**
	 * Constructor for DefaultEigenvalueIterationMonitor. Default is 100000
	 * iterations at most, convergence tolerance of 1e-5 and divergence
	 * tolerance of 1e+5.
	 */
	public DefaultEigenvalueIterationMonitor() {
		this(100000, 1e-5, 1e+5);
	}

	protected boolean convergedI(double r)
		throws IterativeEigenvalueSolverNotConvergedException {

		// Check for convergence
		if (r < ctol)
			return true;

		// Check for divergence
		if (r > dtol)
			throw new IterativeEigenvalueSolverNotConvergedException(
				NotConvergedException.Reason.Divergence,
				this);
		if (iter >= maxIter)
			throw new IterativeEigenvalueSolverNotConvergedException(
				NotConvergedException.Reason.Iterations,
				this);
		if (Double.isNaN(r))
			throw new IterativeEigenvalueSolverNotConvergedException(
				NotConvergedException.Reason.Divergence,
				this);

		// Neither convergence nor divergence
		return false;
	}

}
