/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.test;

import junit.framework.TestCase;
import mt.DenseMatrix;
import mt.Matrix;
import mt.Vector;
import mt.fact.DenseLU;
import mt.test.Utilities;
import mt.util.Matrices;
import smt.CompRowMatrix;
import smt.iter.IterativeSolver;
import smt.iter.IterativeSolverNotConvergedException;

/**
 * Test of the iterative solvers and preconditioners
 */
public abstract class IterativeSolverTest extends TestCase {

	/**
	 * Number of times to repeat tests
	 */
	private int repeat = 5;

	/**
	 * Sizes of the system matrix
	 */
	protected int max = 50, bmax = 10;

	/**
	 * Numerical tolerance
	 */
	protected double tol = 1e-4;

	/**
	 * Diagonal shift for singularity handling
	 */
	protected double shift = 100;

	/**
	 * Square system matrix
	 */
	protected Matrix A;

	/**
	 * Right hand side, right hand for transpose system, and the solution
	 * vector in both cases
	 */
	protected Vector b, bt, x;

	/**
	 * Stores the data of x
	 */
	protected double[] xd;

	/**
	 * Iterative solver to use
	 */
	protected IterativeSolver solver;

	/**
	 * Constructor for IterativeSolverTest
	 */
	public IterativeSolverTest(String arg0) {
		super(arg0);
	}

	protected void setUp() throws Exception {
		createMatrix();

		int n = A.numRows();
		x = Matrices.random(n);
		b = Matrices.random(n);
		bt = Matrices.random(n);

		createSolver();

		// Compute the correct right hand sides
		b = A.mult(x, b);
		bt = A.transMult(x, bt);

		// Store x for later. It is overwritten
		xd = Matrices.getArray(x);

		// Randomize the inital solution vector
		Matrices.random(x);
	}

	protected abstract void createSolver() throws Exception;

	protected void createMatrix() throws Exception {
		// Create an arbitrary matrix
		int n = Utilities.getInt(max), m = Utilities.getInt(Math.min(bmax, n));
		A = new CompRowMatrix(n, n, m);
		Utilities.rowPopulate(A, m);

		// Make it non-singular
		A.addDiagonal(shift);
		DenseLU lu = new DenseLU(new DenseMatrix(A));
		while (lu.isSingular()) {
			A.addDiagonal(shift);
			lu = new DenseLU(new DenseMatrix(A));
		}
	}

	protected void tearDown() throws Exception {
		A = null;
		b = bt = x = null;
		xd = null;
		solver = null;
	}

	public void testSolve() {
		try {
			solver.solve(A, b, x);
			assertSolved();
		} catch (IterativeSolverNotConvergedException e) {
			fail(
				"Solver did not converge: "
					+ e.getReason()
					+ ". Residual="
					+ e.getResidual());
		}
	}

	public void testTransSolve() {
		try {
			solver.transSolve(A, bt, x);
			assertSolved();
		} catch (IterativeSolverNotConvergedException e) {
			fail(
				"Solver did not converge: "
					+ e.getReason()
					+ ". Residual="
					+ e.getResidual());
		}
	}

	public void testRepeatSolve() {
		try {
			for (int i = 0; i < repeat; ++i) {
				solver.solve(A, b, x);
				assertSolved();
				x = Matrices.random(A.numRows());
			}
		} catch (IterativeSolverNotConvergedException e) {
			fail(
				"Solver did not converge: "
					+ e.getReason()
					+ ". Residual="
					+ e.getResidual());
		}
	}

	public void testTransRepeatSolve() {
		try {
			for (int i = 0; i < repeat; ++i) {
				solver.transSolve(A, bt, x);
				assertSolved();
				x = Matrices.random(A.numRows());
			}
		} catch (IterativeSolverNotConvergedException e) {
			fail(
				"Solver did not converge: "
					+ e.getReason()
					+ ". Residual="
					+ e.getResidual());
		}
	}

	public void testMixedRepeatSolve() {
		try {
			for (int i = 0; i < repeat; ++i) {
				solver.solve(A, b, x);
				assertSolved();
				x = Matrices.random(A.numRows());

				solver.transSolve(A, bt, x);
				assertSolved();
				x = Matrices.random(A.numRows());
			}
		} catch (IterativeSolverNotConvergedException e) {
			fail(
				"Solver did not converge: "
					+ e.getReason()
					+ ". Residual="
					+ e.getResidual());
		}
	}

	protected void assertSolved() {
		for (int i = 0; i < xd.length; ++i)
			assertEquals(xd[i], x.get(i), tol);
	}

}
