/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.util;

import java.util.Iterator;
import mt.Iterable;

/**
 * An iterator over an array of iterable objects
 */
public class SuperIterator
	//implements Iterator < SuperIterator.SuperIteratorEntry > {
	implements Iterator {

	private Iterable[] iterable;

	/**
	 * Two iterators. We need the "next" iterator so that hasNext works
	 * properly from one iterable to the next. Using a single iterator won't
	 * cut it
	 */
	private Iterator current, next;

	private int currentIndex, nextIndex;

	/**
	 * Recyled entry returned from next()
	 */
	private SuperIteratorEntry entry;

	/**
	 * Constructor for SuperIterator
	 * 
	 * @param iterable
	 *            Iterable objects to iterate over
	 */
	public SuperIterator(Iterable[] iterable) {
		this.iterable = iterable;
		entry = new SuperIteratorEntry();

		// Try to be somewhat fault tolerant
		if (iterable.length == 0) {
			current = new DummyIterator();
			next = new DummyIterator();
		} else {

			// This moves the next pointer to a non-empty iterable
			next = iterable[nextIndex].iterator();
			moveNext();

			// Then we move the current pointer in the same way
			current = iterable[currentIndex].iterator();
			moveCurrent();

			// Finally, move the next one step ahead if possible
			if (next.hasNext())
				next.next();
		}
	}

	private void moveNext() {
		while (nextIndex < iterable.length - 1 && !next.hasNext())
			next = iterable[++nextIndex].iterator();
	}

	private void moveCurrent() {
		while (currentIndex < iterable.length - 1 && !current.hasNext())
			current = iterable[++currentIndex].iterator();
	}

	public boolean hasNext() {
		return current.hasNext() || next.hasNext();
	}

	//public SuperIteratorEntry next() {
	public Object next() {
		// A wrapped object containing the relevant index and data
		entry.update(currentIndex, current.next());

		// Move current if necessary
		moveCurrent();

		// Move the next pointer
		moveNext();
		if (next.hasNext())
			next.next();

		return entry;
	}

	public void remove() {
		current.remove();
	}

	/**
	 * Dummy iterator, for degenerate cases
	 */
	private class DummyIterator implements Iterator {

		public boolean hasNext() {
			return false;
		}

		public Object next() {
			return null;
		}

		public void remove() {
		}
	}

	/**
	 * Entry returned from this superiterator
	 */
	public class SuperIteratorEntry {

		/**
		 * Index of the iterator which returned this
		 */
		private int i;

		/**
		 * Object returned
		 */
		private Object o;

		void update(int i, Object o) {
			this.i = i;
			this.o = o;
		}

		public int getIndex() {
			return i;
		}

		public Object getObject() {
			return o;
		}

	}

}
