/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Iterator;

import mt.AbstractMatrix;
import mt.DenseVector;
import mt.Matrix;
import mt.MatrixEntry;
import mt.Vector;
import mvio.MatrixInfo;
import mvio.MatrixSize;
import mvio.MatrixVectorReader;

/**
 * Coordinate matrix
 */
public class CoordMatrix extends AbstractMatrix implements Serializable {

	private static final long serialVersionUID = 1093726141416771908L;

	/**
	 * The row indices
	 */
	private int[] rowIndex;

	/**
	 * The column indices
	 */
	private int[] columnIndex;

	/**
	 * The data
	 */
	private double[] data;

	/**
	 * Number of entries used
	 */
	private int used;

	/**
	 * Constructor for CoordMatrix
	 * 
	 * @param in
	 *            Stream to read sparse matrix from
	 */
	public CoordMatrix(InputStream in) throws IOException {
		this(new InputStreamReader(in));
	}

	/**
	 * Constructor for CoordMatrix
	 * 
	 * @param r
	 *            Reader to get sparse matrix from
	 */
	public CoordMatrix(Reader r) throws IOException {
		this(new MatrixVectorReader(r));
	}

	/**
	 * Constructor for CoordMatrix
	 * 
	 * @param r
	 *            Reader to get sparse matrix from
	 */
	public CoordMatrix(MatrixVectorReader r) throws IOException {
		// Start with a zero-sized matrix
		super(0, 0);

		// Get matrix information. Use the header if present, else use a safe
		// default
		MatrixInfo info = null;
		if (r.hasInfo())
			info = r.readMatrixInfo();
		else
			info =
				new MatrixInfo(
					true,
					MatrixInfo.MatrixField.Real,
					MatrixInfo.MatrixSymmetry.General);
		MatrixSize size = r.readMatrixSize(info);

		// Resize the matrix to correct size
		numRows = size.numRows();
		numColumns = size.numColumns();

		// Check that the matrix is in acceptable format
		if (info.isPattern())
			throw new UnsupportedOperationException("Pattern matrices are not supported");
		if (info.isDense())
			throw new UnsupportedOperationException("Dense matrices are not supported");
		if (info.isComplex())
			throw new UnsupportedOperationException("Complex matrices are not supported");

		// Start reading entries
		int[] row = new int[size.numEntries()],
			column = new int[size.numEntries()];
		double[] entry = new double[size.numEntries()];
		r.readCoordinate(row, column, entry);

		// Shift the indices from 1 based to 0 based
		r.add(-1, row);
		r.add(-1, column);

		// In case of symmetry, some extra care is needed
		if (info.isSymmetric() || info.isSkewSymmetric()) {

			// Create the structure
			columnIndex = new int[2 * size.numEntries()];
			rowIndex = new int[2 * size.numEntries()];
			data = new double[2 * size.numEntries()];

			// Insert the entries
			for (int i = 0; i < size.numEntries(); ++i)
				set(row[i], column[i], entry[i]);

			// Put in the other entries
			if (info.isSymmetric())
				for (int i = 0; i < size.numEntries(); ++i)
					set(column[i], row[i], entry[i]);
			else if (info.isSkewSymmetric())
				for (int i = 0; i < size.numEntries(); ++i)
					set(column[i], row[i], -entry[i]);

			// Compact the storage
			compact();
		} else {
			// The unsymmetric case is straight forward
			rowIndex = row;
			columnIndex = column;
			data = entry;
			used = size.numEntries();
		}
	}

	/**
	 * Constructor for CoordMatrix
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of columns
	 * @param nz
	 *            Total number of nonzero entries to preallocate
	 */
	public CoordMatrix(int numRows, int numColumns, int nz) {
		super(numRows, numColumns);
		columnIndex = new int[nz];
		rowIndex = new int[nz];
		data = new double[nz];
	}

	/**
	 * Constructor for CoordMatrix. No preallocation is done
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of columns
	 */
	public CoordMatrix(int numRows, int numColumns) {
		this(numRows, numColumns, 0);
	}

	/**
	 * Constructor for CoordMatrix
	 * 
	 * @param A
	 *            Creates a deep copy of this matrix
	 */
	public CoordMatrix(Matrix A) {
		this(A, true);
	}

	/**
	 * Constructor for CoordMatrix
	 * 
	 * @param A
	 *            Matrix to copy from
	 * @param deep
	 *            True if a deep copy is to be made. If the copy is shallow,
	 *            <code>A</code> must be a <code>CoordMatrix</code>
	 */
	public CoordMatrix(Matrix A, boolean deep) {
		super(A);

		if (deep) {
			int nz = A.cardinality();
			columnIndex = new int[nz];
			rowIndex = new int[nz];
			data = new double[nz];
			set(A);
		} else {
			CoordMatrix Ac = (CoordMatrix) A;
			columnIndex = Ac.getColumnIndices();
			rowIndex = Ac.getRowIndices();
			data = Ac.getData();
			used = data.length;
		}
	}

	/**
	 * Constructor for CoordMatrix
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of columns
	 * @param row
	 *            Row indices
	 * @param column
	 *            Columns indices
	 * @param data
	 *            Numerical entries
	 * @param deep
	 *            True if a deep copy is to be made. Else the matrix uses the
	 *            passed arrays as internal storage
	 */
	public CoordMatrix(
		int numRows,
		int numColumns,
		int[] row,
		int[] column,
		double[] data,
		boolean deep) {
		super(numRows, numColumns);

		used = row.length;
		if (used != column.length || used != data.length)
			throw new IllegalArgumentException("Arrays must be of equal length");

		if (deep) {
			this.rowIndex = (int[]) row.clone();
			this.columnIndex = (int[]) column.clone();
			this.data = (double[]) data.clone();
		} else {
			this.rowIndex = row;
			this.columnIndex = column;
			this.data = data;
		}
	}

	/**
	 * Constructor for CoordMatrix. Creates a deep copy of the passed arrays
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of columns
	 * @param row
	 *            Row indices
	 * @param column
	 *            Columns indices
	 * @param data
	 *            Numerical entries
	 */
	public CoordMatrix(
		int numRows,
		int numColumns,
		int[] row,
		int[] column,
		double[] data) {
		this(numRows, numColumns, row, column, data, true);
	}

	/**
	 * Returns all the column indices
	 */
	public int[] getColumnIndices() {
		compact();
		return columnIndex;
	}

	/**
	 * Returns all the row indices
	 */
	public int[] getRowIndices() {
		compact();
		return rowIndex;
	}

	/**
	 * Returns the matrix data
	 */
	public double[] getData() {
		compact();
		return data;
	}

	public Vector multAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.multAdd(alpha, x, beta, y, z);

		checkMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		// z = beta/alpha * y
		z.set(beta / alpha, y);

		// z = A*x + z
		for (int i = 0; i < used; ++i)
			zd[rowIndex[i]] += data[i] * xd[columnIndex[i]];

		// z = alpha*z = alpha*A*x + beta*y
		return z.scale(alpha);
	}

	public Vector transMultAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.transMultAdd(alpha, x, beta, y, z);

		checkTransMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		// z = beta/alpha * y
		z.set(beta / alpha, y);

		// z = A'x + z
		for (int i = 0; i < used; ++i)
			zd[columnIndex[i]] += data[i] * xd[rowIndex[i]];

		// z = alpha*z = alpha * A'x + beta*y
		return z.scale(alpha);
	}

	public void add(int row, int column, double value) {
		check(row, column);

		int index = getIndex(row, column);
		data[index] += value;
	}

	public double get(int row, int column) {
		check(row, column);

		int index = search(row, column);
		if (index >= 0)
			return data[index];
		else
			return 0.;
	}

	//public Iterator < MatrixEntry > iterator() {
	public Iterator iterator() {
		return new CoordMatrixIterator();
	}

	public void set(int row, int column, double value) {
		check(row, column);

		int index = getIndex(row, column);
		data[index] = value;
	}

	//public CoordMatrix zero() {
	public Matrix zero() {
		Arrays.fill(data, 0);
		return this;
	}

	//public CoordMatrix copy() {
	public Matrix copy() {
		return new CoordMatrix(this);
	}

	/**
	 * Tries to store the matrix as compactly as possible
	 */
	public void compact() {
		int nz = cardinality();

		if (nz < data.length) {
			int[] newRow = new int[nz], newColumn = new int[nz];
			double[] newData = new double[nz];

			// Copy only non-zero entries
			for (int i = 0, j = 0; i < used; ++i)
				if (data[i] != 0.) {
					newRow[j] = rowIndex[j];
					newColumn[j] = columnIndex[j];
					newData[j] = data[j];
					j++;
				}

			rowIndex = newRow;
			columnIndex = newColumn;
			data = newData;
			used = nz;
		}
	}

	/**
	 * Tries to get the index, or allocates more space if needed
	 */
	private int getIndex(int rowI, int columnI) {
		int index = search(rowI, columnI);
		if (index >= 0) // Found
			return index;
		else { // Not found

			// Need to allocate more space
			if (used >= data.length)
				if (used == 0)
					realloc(1);
				else
					realloc(2 * used);

			// Put in the new indices
			rowIndex[used] = rowI;
			columnIndex[used] = columnI;
			return used++;
		}
	}

	/**
	 * Reallocates the internal arrays to a new size
	 */
	private void realloc(int newSize) {
		int[] newRow = new int[newSize], newColumn = new int[newSize];
		double[] newData = new double[newSize];

		System.arraycopy(rowIndex, 0, newRow, 0, used);
		System.arraycopy(columnIndex, 0, newColumn, 0, used);
		System.arraycopy(data, 0, newData, 0, used);

		rowIndex = newRow;
		columnIndex = newColumn;
		data = newData;
	}

	/**
	 * Searches for the given entry. Returns -1 if it is not found
	 */
	private int search(int rowI, int columnI) {
		for (int i = 0; i < used; ++i)
			if (rowIndex[i] == rowI && columnIndex[i] == columnI)
				return i;
		return -1;
	}

	/**
	 * Iterator over a coordinate matrix
	 */
	private class CoordMatrixIterator extends AbstractMatrixIterator {

		private int cursor, cursorNext;

		public CoordMatrixIterator() {
			entry = new CoordMatrixEntry();
			init();
		}

		public boolean hasNext() {
			return cursor < data.length;
		}

		protected boolean hasNextNext() {
			return cursorNext < data.length;
		}

		protected void cycle() {
			super.cycle();
			cursor = cursorNext;
		}

		protected void updateEntry() {
			((CoordMatrixEntry) entry).update(
				rowIndex[cursor],
				columnIndex[cursor],
				data[cursor],
				cursor);
		}

		protected double nextValue() {
			return data[cursorNext];
		}

		protected void nextPosition() {
			cursorNext++;
			if (cursorNext < data.length) {
				rowNext = rowIndex[cursorNext];
				columnNext = columnIndex[cursorNext];
			} else {
				rowNext = numRows();
				columnNext = numColumns();
			}
		}

	}

	/**
	 * Entry returned when iterating over this matrix
	 */
	private class CoordMatrixEntry extends RefMatrixEntry {

		private int cursor;

		public void update(int row, int column, double value, int cursor) {
			super.update(row, column, value);
			this.cursor = cursor;
		}

		public void set(double value) {
			this.value = value;
			data[cursor] = value;
		}

	}

}
