/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

//import java.util.Formatter;
import java.util.Iterator;

/**
 * Partial implementation of <code>Vector</code>. Subclasses should provide
 * implementations of the following methods:
 * <ul>
 * <li>Basic operations:</li>
 * <ul>
 * <li>{@link #add(int, double) add}</li>
 * <li>{@link #get(int, ) get}</li>
 * <li>{@link #set(int, double) set}</li>
 * <li>{@link #copy() copy}</li>
 * </ul>
 * <li>Kernel mathematical operations. Default implementations are provided,
 * but subclasses can provide faster version. Kernel operations are:</li>
 * <ul>
 * <li>{@link #set(double, mt.Vector, double, mt.Vector) set}</li>
 * <li>{@link #dot(mt.Vector) dot}</li>
 * <li>{@link #scale(double) scale}</li>
 * </ul>
 * <li>Faster iterators. Most other methods in the class uses an iterator, and
 * an efficient iterator can yield substantial improvements in performance. The
 * default iterator simply calls {@link #get(int) get}.</li>
 * </ul>
 */
public abstract class AbstractVector implements Vector {

	/**
	 * Size of the vector
	 */
	protected int size;

	/**
	 * Constructor for AbstractVector.
	 * 
	 * @param size
	 *            Size of the vector
	 */
	protected AbstractVector(int size) {
		if (size < 0)
			throw new IllegalArgumentException("Vector size cannot be negative");
		this.size = size;
	}

	/**
	 * Constructor for AbstractVector, same size as x
	 * 
	 * @param x
	 *            Vector to get the size from
	 */
	protected AbstractVector(Vector x) {
		this.size = x.size();
	}

	public int size() {
		return size;
	}

	public void add(int index, double value) {
		set(index, value + get(index));
	}

	public void add(int[] index, double[] values) {
		check(index, values);

		for (int i = 0; i < index.length; ++i)
			add(index[i], values[i]);
	}

	public double get(int index) {
		throw new UnsupportedOperationException();
	}

	public double[] get(int[] index, double[] values) {
		check(index, values);

		for (int i = 0; i < index.length; ++i)
			values[i] = get(index[i]);
		return values;
	}

	public void set(int index, double value) {
		throw new UnsupportedOperationException();
	}

	public void set(int[] index, double[] values) {
		check(index, values);

		for (int i = 0; i < index.length; ++i)
			set(index[i], values[i]);
	}

	/**
	 * Checks the index
	 */
	protected void check(int index) {
		if (index < 0)
			throw new IllegalArgumentException("index is negative");
		if (index >= size)
			throw new IllegalArgumentException("index >= size");
	}

	/**
	 * Checks the indices
	 */
	protected void check(int[] index) {
		for (int i = 0; i < index.length; ++i)
			check(index[i]);
	}

	/**
	 * Checks indices and values
	 */
	protected void check(int[] index, double[] values) {
		if (index.length != values.length)
			throw new IllegalArgumentException("index.length != values.length");
		check(index);
	}

	public Vector copy() {
		throw new UnsupportedOperationException();
	}

	public Vector zero() {
		//for (VectorEntry e : this)
		//	e.set(0);
		VectorEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			e.set(0);
		}
		return this;
	}

	public Iterator iterator() {
		return new RefVectorIterator();
	}

	public String toString() {
		// Output into coordinate format. Indices start from 1 instead of 0
		//Formatter out = new Formatter();
		//out.format("%10d %19d\n", size, cardinality());
		String s = Integer.toString(size) + " " + cardinality() + "\n";
		//for (VectorEntry e : this)
		//	s += (e.index() + 1) + " " + e.get() + "\n";
		VectorEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			s += (e.index() + 1) + " " + e.get() + "\n";
		}
		return s;
	}

	public Vector set(double alpha, Vector y, double beta, Vector z) {
		checkSet(y, z);

		for (int i = 0; i < size; ++i)
			set(i, alpha * y.get(i) + beta * z.get(i));

		return this;
	}

	/**
	 * Checks for conformance in vector set
	 */
	protected void checkSet(Vector y, Vector z) {
		if (size != y.size())
			throw new IllegalArgumentException("x.size() != y.size()");
		if (y.size() != z.size())
			throw new IllegalArgumentException("y.size() != z.size()");
	}

	public Vector add(double alpha, Vector y, double beta) {
		return set(alpha, y, beta, this);
	}

	public Vector set(double alpha, Vector y, Vector z) {
		return set(alpha, y, 1, z);
	}

	public Vector add(double alpha, Vector y) {
		return add(alpha, y, 1);
	}

	public Vector set(Vector y, double beta, Vector z) {
		return set(1, y, beta, z);
	}

	public Vector add(Vector y, double beta) {
		return add(1, y, beta);
	}

	public Vector set(Vector y, Vector z) {
		return set(1, y, 1, z);
	}

	public Vector add(Vector y) {
		return add(1, y, 1);
	}

	public double dot(Vector y) {
		checkSize(y);

		double ret = 0;
		//for (VectorEntry e : this)
		//	ret += e.get() * y.get(e.index());
		VectorEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			ret += e.get() * y.get(e.index());
		}
		
		return ret;
	}

	/**
	 * Checks for conformant sizes
	 */
	protected void checkSize(Vector y) {
		if (size != y.size())
			throw new IllegalArgumentException("x.size() != y.size()");
	}

	public double norm(Norm type) {
		if (type == Norm.One)
			return norm1();
		else if (type == Norm.Two)
			return norm2();
		else if (type == Norm.TwoRobust)
			return norm2_robust();
		else // Infinity
			return normInf();
	}

	protected double norm1() {
		double sum = 0;
		/*for (VectorEntry e : this)
			sum += Math.abs(e.get());*/
		VectorEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			sum += Math.abs(e.get());
		}
		return sum;
	}

	protected double norm2() {
		double norm = 0;
		/*for (VectorEntry e : this)
			norm += e.get() * e.get();*/
		VectorEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			norm += e.get() * e.get();
		}
		return Math.sqrt(norm);
	}

	protected double norm2_robust() {
		double scale = 0, ssq = 1;
		/*for (VectorEntry e : this) {
			double xval = e.get();
			if (xval != 0) {
				double absxi = Math.abs(xval);
				if (scale < absxi) {
					ssq = 1 + ssq * Math.pow(scale / absxi, 2);
					scale = absxi;
				} else
					ssq = ssq + Math.pow(absxi / scale, 2);
			}
		}*/
		VectorEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			double xval = e.get();
			if (xval != 0) {
				double absxi = Math.abs(xval);
				if (scale < absxi) {
					ssq = 1 + ssq * Math.pow(scale / absxi, 2);
					scale = absxi;
				} else
					ssq = ssq + Math.pow(absxi / scale, 2);
			}
		}
		return scale * Math.sqrt(ssq);
	}

	protected double normInf() {
		double max = 0;
		/*for (VectorEntry e : this)
			max = Math.max(Math.abs(e.get()), max);*/
		VectorEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			max = Math.max(Math.abs(e.get()), max);
		}
		return max;
	}

	public Vector scale(double alpha) {
		// Quick return if possible
		if (alpha == 0)
			return zero();
		else if (alpha == 1)
			return this;

		/*for (VectorEntry e : this)
			e.set(alpha * e.get());*/
		VectorEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			e.set(alpha * e.get());
		}

		return this;
	}

	public Vector set(double alpha, Vector y) {
		checkSize(y);

		if (this == y)
			return scale(alpha);
		else if (alpha == 0)
			return zero();

		zero();
		/*for (VectorEntry e : y)
			set(e.index(), alpha * e.get());*/
		VectorEntry e;
		Iterator iter = y.iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			set(e.index(), alpha * e.get());
		}
		return this;
	}

	public Vector set(Vector y) {
		checkSize(y);

		if (this == y)
			return this;

		zero();
		/*for (VectorEntry e : y)
			set(e.index(), e.get());*/
		VectorEntry e;
		Iterator iter = y.iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			set(e.index(), e.get());
		}
		return this;
	}

	public int cardinality() {
		int nz = 0;
		//for (VectorEntry e : this)
		//	nz++;
		for(Iterator iter = iterator(); iter.hasNext(); iter.next(), ++nz) {}
		return nz;
	}

	/**
	 * Partial implementation of a vector iterator. Subclasses must implement
	 * all abstract methods, and call <code>init</code> as the last step of
	 * the constructor. The entry must also be set.
	 */
	protected abstract class AbstractVectorIterator
		//implements Iterator < VectorEntry > {
		implements Iterator {

		/**
		 * Current cursor index
		 */
		protected int cursor;

		/**
		 * Next cursor index
		 */
		protected int cursorNext;

		/**
		 * Referenced entry
		 */
		protected VectorEntry entry;

		/**
		 * Sets initial positions. Must be called before the iterator is used
		 */
		protected void init() {
			// Position at next non-zero entry if necessary
			if (hasNextNext() && nextValue() == 0)
				rePositionNext();

			// Use that as current entry
			cycle();

			// Then find the next one. Needed when the user calls next()
			if (hasNextNext())
				rePositionNext();
		}

		public boolean hasNext() {
			return cursor < size;
		}

		public void remove() {
			entry.set(0);
		}

		//public VectorEntry next() {
		public Object next() {
			// Sets the entry to the current position
			updateEntry();

			// Sets position to the next
			cycle();

			// Finds the next position
			rePositionNext();

			return entry;
		}

		/**
		 * Cycle the next position into the current
		 */
		protected void cycle() {
			cursor = cursorNext;
		}

		/**
		 * Sets the entry to the current position
		 */
		protected abstract void updateEntry();

		/**
		 * Moves the next-pointers to the next non-zero position
		 */
		protected void rePositionNext() {
			do {
				nextPosition();
			} while (hasNextNext() && nextValue() == 0);
		}

		/**
		 * Returns true if the next position is within bounds
		 */
		protected boolean hasNextNext() {
			return cursorNext < size;
		}

		/**
		 * Moves to the next entry, irrespective of wheter it's non-zero. It
		 * may move outside the bounds of the vector only if there are no more
		 * entries left
		 */
		protected abstract void nextPosition();

		/**
		 * Returns value at the next position
		 */
		protected double nextValue() {
			return get(cursorNext);
		}

	}

	/**
	 * Iterator over a general vector. References the underlying vector using
	 * get and set operations
	 */
	protected class RefVectorIterator extends AbstractVectorIterator {

		/**
		 * Constructor for RefVectorIterator
		 */
		public RefVectorIterator() {
			entry = new RefVectorEntry();
			init();
		}

		protected void updateEntry() {
			((RefVectorEntry) entry).update(cursor, get(cursor));
		}

		protected void nextPosition() {
			cursorNext++;
		}

	}

	/**
	 * Vector entry backed by the vector. May be recycled for higher performance
	 */
	protected class RefVectorEntry implements VectorEntry {

		protected int index;

		protected double value;

		public void update(int index, double value) {
			this.index = index;
			this.value = value;
		}

		public int index() {
			return index;
		}

		public double get() {
			return value;
		}

		public void set(double value) {
			this.value = value;
			AbstractVector.this.set(index, value);
		}
	}

	public Vector set(double alpha) {
		if (alpha == 0)
			return zero();

		for (int i = 0; i < size; ++i)
			set(i, alpha);
		return this;
	}

}
