/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import mt.ll.BLASkernel.Diag;

import java.util.Iterator;

/**
 * Unit upper triangular packed matrix. Same storage as {@link mt.UpperTriangPackMatrix UpperTriangPackMatrix},
 * but the main diagonal is assumed to be all ones.
 */
public class UnitUpperTriangPackMatrix extends UpperTriangPackMatrix {

	private static final long serialVersionUID = 987016325527263111L;

	/**
	 * Constructor for UnitUpperTriangPackMatrix
	 * 
	 * @param n
	 *            Size of the matrix. Since the matrix must be square, this
	 *            equals both the number of rows and columns
	 */
	public UnitUpperTriangPackMatrix(int n) {
		super(n, Diag.Unit);
	}

	/**
	 * Constructor for UnitUpperTriangPackMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from. Only the entries of the
	 *            relevant part are copied
	 */
	public UnitUpperTriangPackMatrix(Matrix A) {
		this(A, true);
	}

	/**
	 * Constructor for UnitUpperTriangPackMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from. Only the entries of the
	 *            relevant part are copied
	 * @param deep
	 *            True if the copy is deep, else false (giving a shallow copy).
	 *            For shallow copies, <code>A</code> must be a packed matrix
	 */
	public UnitUpperTriangPackMatrix(Matrix A, boolean deep) {
		super(A, deep, Diag.Unit);
	}

	public void add(int row, int column, double value) {
		if (row == column)
			throw new IllegalArgumentException("row == column");
		super.add(row, column, value);
	}

	public double get(int row, int column) {
		if (row == column)
			return 1;
		return super.get(row, column);
	}

	public void set(int row, int column, double value) {
		if (row == column)
			throw new IllegalArgumentException("row == column");
		super.set(row, column, value);
	}

	void copy(Matrix A) {
		/*for (MatrixEntry e : A)
			if (e.row() < e.column())
				set(e.row(), e.column(), e.get());*/
		MatrixEntry e;
		Iterator iter = A.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			if (e.row() < e.column())
				set(e.row(), e.column(), e.get());
		}
	}

	//public UnitUpperTriangPackMatrix copy() {
	public Matrix copy() {
		return new UnitUpperTriangPackMatrix(this);
	}

}
