/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt;

import mt.DenseVector;
import mt.Matrix;
import mt.Vector;

/**
 * Compressed row storage (CRS) matrix built from sparse vectors
 */
public class FlexCompRowMatrix extends AbstractRowMatrix {

	/**
	 * Constructor for FlexCompRowMatrix
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of column
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 */
	/*public FlexCompRowMatrix(int numRows, int numColumns, int numThreads) {
		super(numRows, numColumns, new SparseVector(numColumns), numThreads);
	}*/

	/**
	 * Constructor for FlexCompRowMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 * @param deep
	 *            True for a deep copy, false for a reference copy. A reference
	 *            copy can only be made of a <code>FlexCompRowMatrix</code>
	 */
	/*public FlexCompRowMatrix(Matrix A, int numThreads, boolean deep) {
		super(A, new SparseVector(A.numColumns()), numThreads, deep);
	}*/

	/**
	 * Constructor for FlexCompRowMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from. The copy will be deep
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 */
	/*public FlexCompRowMatrix(Matrix A, int numThreads) {
		super(A, new SparseVector(A.numColumns()), numThreads);
	}*/

	/**
	 * Constructor for FlexCompRowMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of column
	 */
	public FlexCompRowMatrix(int numRows, int numColumns) {
		super(numRows, numColumns, new SparseVector(numColumns));
	}

	/**
	 * Constructor for FlexCompRowMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param A
	 *            Matrix to copy contents from. The copy will be deep
	 */
	public FlexCompRowMatrix(Matrix A) {
		super(A, new SparseVector(A.numColumns()));
	}

	/**
	 * Constructor for FlexCompRowMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param A
	 *            Matrix to copy contents from
	 * @param deep
	 *            True for a deep copy, false for a reference copy. A reference
	 *            copy can only be made of a <code>FlexCompRowMatrix</code>
	 */
	public FlexCompRowMatrix(Matrix A, boolean deep) {
		super(A, new SparseVector(A.numColumns()), deep);
	}

	public Vector transMultAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.transMultAdd(alpha, x, beta, y, z);

		checkTransMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		// z = beta/alpha * y
		z.set(beta / alpha, y);

		// z = A'x + z
		for (int i = 0; i < numRows; ++i) {
			SparseVector v = (SparseVector) rowD[i];
			int[] index = v.getIndex();
			double[] data = v.getData();
			int length = v.used();
			for (int j = 0; j < length; ++j)
				zd[index[j]] += data[j] * xd[i];
		}

		// z = alpha*z = alpha * A'x + beta*y
		return z.scale(alpha);
	}

	//public FlexCompRowMatrix copy() {
	public Matrix copy() {
		return new FlexCompRowMatrix(this);
	}

	//public SparseVector getRow(int i) {
	public Vector getRow(int i) {
		return (SparseVector) rowD[i];
	}

	/**
	 * Sets the given row equal the passed vector
	 */
	public void setRow(int i, SparseVector x) {
		if (x.size() != numColumns)
			throw new IllegalArgumentException("New row must be of the same size as existing row");
		rowD[i] = x;
	}

	/**
	 * Tries to store the matrix as compactly as possible
	 */
	public void compact() {
		/*for (Vector v : rowD)
			 ((SparseVector) v).compact();*/
		for(int i = 0; i < rowD.length; ++i) {
			((SparseVector) rowD[i]).compact();
		}
	}

}
