/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.mixed;

import mt.Matrix;
import mt.Vector;
import smt.iter.IterativeSolverNotConvergedException;

/**
 * Uzawa algorithm for mixed problems. Uses a fixed steplength, which by
 * default is <code>1</code>.
 */
public class Uzawa extends AbstractMixedSolver {

	/**
	 * Steplength
	 */
	private double alpha;

	/**
	 * Temporary work vectors
	 */
	private Vector qr, r, t;

	/**
	 * Constructor for Uzawa. Uses a steplength of 1
	 * 
	 * @param qTemplate
	 *            Vector of size <i>m</i>, used as template in preallocating
	 *            work vectors
	 * @param uTemplate
	 *            Vector of size <i>n</i>, used as template in preallocating
	 *            work vectors
	 */
	public Uzawa(Vector qTemplate, Vector uTemplate) {
		this(qTemplate, uTemplate, 1);
	}

	/**
	 * Constructor for Uzawa
	 * 
	 * @param qTemplate
	 *            Vector of size <i>m</i>, used as template in preallocating
	 *            work vectors
	 * @param uTemplate
	 *            Vector of size <i>n</i>, used as template in preallocating
	 *            work vectors
	 * @param alpha
	 *            Fixed steplength to use in the descent process
	 */
	public Uzawa(Vector qTemplate, Vector uTemplate, double alpha) {
		super(qTemplate);
		setSteplength(alpha);

		qr = qTemplate.copy();

		r = uTemplate.copy();
		t = uTemplate.copy();
	}

	/**
	 * Sets the steplength
	 *
	 * @param alpha
	 *            New steplength to use. Must be positive
	 */
	public void setSteplength(double alpha) {
		this.alpha = alpha;
		if (alpha <= 0)
			throw new IllegalArgumentException("alpha must be positive");
	}

	protected void solveI(
		Matrix A,
		Matrix B,
		Matrix Bt,
		Matrix C,
		Vector q,
		Vector u,
		Vector f,
		Vector g)
		throws IterativeSolverNotConvergedException {

		C.multAdd(u, -1., g, t); // t = C*u - g
		B.multAdd(q, t, r); // r = B*q + t

		for (iter.setFirst(); !iter.converged(r); iter.next()) {
			Bt.multAdd(-1., u, f, qr); // qr = f - Bt*u
			solver.solve(A, qr, q); // q = A\qr

			C.multAdd(u, -1., g, t); // t = C*u - g
			B.multAdd(q, t, r); // r = B*q + t
			u.add(alpha, r); // u = u + alpha*r
		}
	}

}
