/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

/**
 * Basic vector interface. It holds <code>double</code>s in some array, and is
 * used alongside {@link mt.Matrix Matrix} in numerical computations.
 * 
 * <h4>Basic operations</h4>
 * <p>
 * To query the size of a vector, simply use the method {@link #size() size}.
 * </p>
 * <p>
 * The simplest method to assemble a vector is to use the <i>set</i>,<i>
 * get</i> and <i>add</i> methods. These all have two variants; scalar and
 * blocked, where the blocked operations typically have higher performance.
 * Note that indices are zero-based (typical for Java and C). This means that
 * the indices range from 0 to <i>size-1</i>. It is legal to have <i>size
 * </i> equal zero.
 * </p>
 * <p>
 * Other basic operations are {@link #zero() zero} which zeros all the entries
 * of the vector, which can be cheaper than either zeroing the vecot manually,
 * or creating a new vector, and the operation {@link #copy() copy} which
 * creates a deep copy of the vector. This copy has separate storage, but
 * starts with the same contents as the calling vector.
 * </p>
 * 
 * <h4>Iterators</h4>
 * <p>
 * The vector interface extends <code>Iterable</code>, and the iterator
 * returns a {@link mt.VectorEntry VectorEntry} which contains current index
 * and entry value. The iterator only passes over non-zero entries, and may
 * skip explicit zeros. Using an iterator, many simple and efficient algorithms
 * can be created. The iterator also permits changing values in the vector,
 * however only non-zero entries can be changed.
 * </p>
 * 
 * <h4>Basic linear algebra</h4>
 * <p>
 * A selection of basic linear algebra operations are available. To ensure high
 * efficiency, little or no internal memory allocation is done, and the user is
 * required to supply the output arguments.
 * </p>
 * <p>
 * The operations available include:
 * </p>
 * <dl>
 * <dt><i>Additions</i></dt>
 * <dd>Vectors can be added to each other, even if their underlying matrix
 * structures are incompatible</dd>
 * <dt><i>Norms</i></dt>
 * <dd>Both innerproducts and norms can be computed. Several common norms are
 * supported</dd>
 * </dl>
 * 
 * <h4>Thread safety</h4>
 * <p>
 * A vector is not thread safe by default. Use
 * {@link mt.util.Matrices.synchronizedVector(mt.Vector)}
 * to create a synchronized wrapper. Note that this only wraps the basic
 * assembly operations (<i>set</i> and <i>add</i>). Algebraic operations
 * and iterators which change the vector are not thread safe.
 * </p>
 */
public interface Vector extends Iterable {

	/**
	 * Size of the vector
	 */
	int size();

	/**
	 * <code>x(index) += value</code>
	 */
	void add(int index, double value);

	/**
	 * <code>x(index) = value</code>
	 */
	void set(int index, double value);

	/**
	 * Returns <code>x(index)</code>
	 */
	double get(int index);

	/**
	 * <code>x(index) += values</code>, blockwise
	 */
	void add(int[] index, double[] values);

	/**
	 * <code>x(index) = values</code>, blockwise
	 */
	void set(int[] index, double[] values);

	/**
	 * Returns the block <code>x(index)</code> into the supplied array
	 */
	double[] get(int[] index, double[] values);

	/**
	 * Creates a deep copy of the vector
	 */
	Vector copy();

	/**
	 * Zeros all the entries in the vector, while preserving any underlying
	 * structure
	 */
	Vector zero();

	/**
	 * <code>x=alpha*x</code>
	 * 
	 * @return x
	 */
	Vector scale(double alpha);

	/**
	 * <code>x=y</code>
	 * 
	 * @return x
	 */
	Vector set(Vector y);

	/**
	 * <code>x=alpha*y</code>
	 * 
	 * @return x
	 */
	Vector set(double alpha, Vector y);

	/**
	 * <code>x = alpha*y + beta*z</code>
	 * 
	 * @return x
	 */
	Vector set(double alpha, Vector y, double beta, Vector z);

	/**
	 * <code>x = y + beta*z</code>
	 * 
	 * @return x
	 */
	Vector set(Vector y, double beta, Vector z);

	/**
	 * <code>x = alpha*y + z</code>
	 * 
	 * @return x
	 */
	Vector set(double alpha, Vector y, Vector z);

	/**
	 * <code>x = y + z</code>
	 * 
	 * @return x
	 */
	Vector set(Vector y, Vector z);

	/**
	 * <code>x = alpha*y + beta*x</code>
	 * 
	 * @return x
	 */
	Vector add(double alpha, Vector y, double beta);

	/**
	 * <code>x = y + beta*x</code>
	 * 
	 * @return x
	 */
	Vector add(Vector y, double beta);

	/**
	 * <code>x = alpha*y + x</code>
	 * 
	 * @return x
	 */
	Vector add(double alpha, Vector y);

	/**
	 * <code>x = y + x</code>
	 * 
	 * @return x
	 */
	Vector add(Vector y);

	/**
	 * <code>x<sup>T</sup>*y</code>
	 */
	double dot(Vector y);

	/**
	 * Returns the number of non-zero entries in the vector
	 */
	int cardinality();

	/**
	 * Sets this vector equal the given scalar
	 *
	 * @return x
	 */
	Vector set(double alpha);

	/**
	 * Computes the given norm of the vector
	 * 
	 * @param type
	 *            The type of norm to compute
	 */
	double norm(Norm type);

	/**
	 * Supported vector-norms. The difference between the two 2-norms is
	 * that one is fast, but can overflow, while the robust version is
	 * overflow resitant, but slower.
	 */
	//enum Norm {
		
		/**
		 * Sum of the absolute values of the entries
		 */
		//One,
		
		/**
		 * The root of sum of squares
		 */
		//Two,
		
		/**
		 * As the 2 norm may overflow, an overflow resistant version is also
		 * available. Note that it may be slower.
		 */
		//TwoRobust,
		
		/**
		 * Largest entry in absolute value
		 */
		//Infinity };
	
	static class Norm {
		public static final Norm One = new Norm();
		public static final Norm Two = new Norm();
		public static final Norm TwoRobust = new Norm();
		public static final Norm Infinity = new Norm();
		
		private Norm() {}
	}

}
