/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.test;

import junit.framework.TestCase;
import mt.Matrix;
import mt.Vector;
import mt.fact.NotConvergedException;
import mt.fact.SymmDenseEigenvalueComputer;
import mt.fact.SymmEigenvalueDecomposition;
import mt.test.Utilities;
import mt.util.Matrices;
import smt.CompRowMatrix;
import smt.iter.eig.DefaultEigenvalueIterationMonitor;
import smt.iter.eig.IterativeEigenvalueSolver;
import smt.iter.eig.IterativeEigenvalueSolverNotConvergedException;

/**
 * Test of the iterative eigenvalue solvers
 */
public abstract class IterativeEigenvalueSolverTest extends TestCase {

	/**
	 * Sizes of the system matrix
	 */
	protected int max = 10, bmax = 5;

	/**
	 * Numerical tolerance
	 */
	protected double tol = 1e-3;

	/**
	 * Square system matrix
	 */
	protected Matrix A;

	protected Vector x;

	/**
	 * Iterative solver to use
	 */
	protected IterativeEigenvalueSolver solver;

	public IterativeEigenvalueSolverTest(String arg0) {
		super(arg0);
	}

	protected void setUp() throws Exception {
		int n = Utilities.getInt(max), m = Utilities.getInt(Math.min(bmax, n));
		A = new CompRowMatrix(n, n, m);
		x = Matrices.random(n);
		Utilities.symmetryPopulate(A, m);

		// Create actual solver
		createSolver();

		solver.setIterationMonitor(
			new DefaultEigenvalueIterationMonitor(10000, tol, 1e+5));
	}

	protected abstract void createSolver() throws Exception;

	protected void tearDown() throws Exception {
		A = null;
		x = null;
		solver = null;
	}

	public void testSolve() {
		// Create some random initial eigenvector and eigenvalue
		int n = A.numRows();
		double v = Math.random();

		try {
			v = solver.solve(A, v, x);

			// Verify that A*x = v*x
			Vector r = x.copy().zero();
			assertEquals(
				0,
				A.multAdd(x, -v, x, r).norm(Vector.Norm.Infinity),
				tol * 1e+1);

			// Verify that the eigenvalue exists
			SymmEigenvalueDecomposition evd =
				SymmDenseEigenvalueComputer.factorize(A);
			double[] rv = evd.getEigenvalues();
			assertEquals(rv, v);
		} catch (IterativeEigenvalueSolverNotConvergedException e) {
			fail(
				"Solver did not converge: "
					+ e.getReason()
					+ ". Residual="
					+ e.getResidual());
		} catch (NotConvergedException e) {
			fail("LAPACK eigenvalue solver did not converge");
		}
	}

	protected void assertEquals(double[] rv, double v) {
		for (int i = 0; i < rv.length; ++i)
			if (Math.abs(rv[i] - v) < tol * 1e+1)
				return;
		fail("No match for the eigenvalue");
	}

}
