/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.eig;

import mt.Matrix;
import mt.Vector;
import mt.Vector.Norm;
import mt.VectorEntry;

import java.util.Iterator;

/**
 * Partial implementation of IterativeEigenvalueSolver
 */
public abstract class AbstractIterativeEigenvalueSolver
	implements IterativeEigenvalueSolver {

	/**
	 * Monitors for convergence and divergence
	 */
	protected EigenvalueIterationMonitor iter;

	/**
	 * Transformation
	 */
	protected EigenvalueTransformation et;

	/**
	 * Constructor for AbstractIterativeEigenvalueSolver
	 */
	public AbstractIterativeEigenvalueSolver() {
		iter = new DefaultEigenvalueIterationMonitor();
		et = new NoEigenvalueTransformation();
	}

	public double solve(Matrix A, double v, Vector x)
		throws IterativeEigenvalueSolverNotConvergedException {
		checkSolveArguments(A, x);

		// Normalize the eigenvector. If the norm is near zero, it is randomized
		double norm = x.norm(Norm.Two);
		if (Math.abs(norm) < 1e-12)
			randomize(x);
		else
			x.scale(1. / norm);

		return solveI(A, v, x);
	}

	protected abstract double solveI(Matrix A, double v, Vector x)
		throws IterativeEigenvalueSolverNotConvergedException;

	/**
	 * Checks that all sizes conform
	 */
	private void checkSolveArguments(Matrix A, Vector x) {
		if (!A.isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (x.size() != A.numRows())
			throw new IllegalArgumentException("x.size() != A.numRows()");
	}

	public void setEigenvalueTransformation(EigenvalueTransformation st) {
		this.et = st;
	}

	public EigenvalueTransformation getEigenvalueTransformation() {
		return et;
	}

	public void setIterationMonitor(EigenvalueIterationMonitor iter) {
		this.iter = iter;
	}

	public EigenvalueIterationMonitor getIterationMonitor() {
		return iter;
	}

	protected void randomize(Vector x) {
		x.set(1);
		/*for (VectorEntry e : x)
			e.set(Math.random());*/
		VectorEntry e;
		Iterator iter = x.iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			e.set(Math.random());
		}
		
		x.scale(1. / x.norm(Norm.Two));
	}
}
