package org.baderlab.brain.correlationlearn;

import mt.MatrixEntry;
import smt.FlexCompRowMatrix;

/**
 * Copyright (c) 2005 Memorial Sloan-Kettering Cancer Center
 * *
 * * Code written by: Gary Bader
 * * Authors: Gary Bader, Chris Sander
 * *
 * * This library is free software; you can redistribute it and/or modify it
 * * under the terms of the GNU Lesser General Public License as published
 * * by the Free Software Foundation; either version 2.1 of the License, or
 * * any later version.
 * *
 * * This library is distributed in the hope that it will be useful, but
 * * WITHOUT ANY WARRANTY, WITHOUT EVEN THE IMPLIED WARRANTY OF
 * * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * * documentation provided hereunder is on an "as is" basis, and
 * * Memorial Sloan-Kettering Cancer Center
 * * has no obligations to provide maintenance, support,
 * * updates, enhancements or modifications.  In no event shall the
 * * Memorial Sloan-Kettering Cancer Center
 * * be liable to any party for direct, indirect, special,
 * * incidental or consequential damages, including lost profits, arising
 * * out of the use of this software and its documentation, even if
 * * Memorial Sloan-Kettering Cancer Center
 * * has been advised of the possibility of such damage.  See
 * * the GNU Lesser General Public License for more details.
 * *
 * * You should have received a copy of the GNU Lesser General Public License
 * * along with this library; if not, write to the Free Software Foundation,
 * * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * *
 * * User: GaryBader
 * * Date: Nov 1, 2005
 * * Time: 6:07:33 PM
 */

/**
 * Handles the storage of a protein-protein residue correlation matrix
 */
public class ResidueResidueCorrelationMatrix
{
    /**
     * The actual protein-protein residue correlation matrix
     */
    protected FlexCompRowMatrix correlationMatrix = null;

    /**
     * Optimization variable for sparse matrix
     */
    protected boolean domainPositionsAsRows = false;

    /**
     * The number of rows in the correlation matrix
     */
    protected int numRows = 0;

    /**
     * The number of columns in the correlation matrix
     */
    protected int numColumns = 0;

    /**
     * Creates an instance of the actual protein-protein residue correlation matrix
     * @param numDomainPositionsPerFeature  The number of domain positions per feature
     * @param numPeptidePositionsPerFeature The number of peptide positions per feature
     * @param totalDomainSequenceLength The total domain sequence length
     * @param totalPeptideSequenceLength The total peptide sequence length
     */
    public ResidueResidueCorrelationMatrix(int numDomainPositionsPerFeature, int numPeptidePositionsPerFeature, int totalDomainSequenceLength, int totalPeptideSequenceLength)
    {
        int maxDomainIndex = FeatureUtils.getMaxIndex(numDomainPositionsPerFeature, totalDomainSequenceLength);
        int maxPeptideIndex = FeatureUtils.getMaxIndex(numPeptidePositionsPerFeature, totalPeptideSequenceLength);

        //we want to keep the number of rows low, because each one is a SparseVector
        //note - for some sparse matrix data structures, like RowCompressed, you may have to switch these based on size
        if (maxDomainIndex > maxPeptideIndex) {
            domainPositionsAsRows = false;
            this.numRows = maxPeptideIndex;
            this.numColumns = maxDomainIndex;
        } else {
            domainPositionsAsRows = true;
            this.numRows = maxDomainIndex;
            this.numColumns = maxPeptideIndex;
        }

        //create a sparse 2D matrix for storing the correlations.  The matrix is conceptually 2 + the total number of
        //positions considered dimensions, but this is dealt with by computing an index to map from higher D to 2D.
        correlationMatrix = new FlexCompRowMatrix(numRows, numColumns);
    }

    /**
     * Add a single correlation count to the protein-protein residue correlation matrix
     * @param domainFeature  The domain feature to address in the matrix
     * @param peptideFeature The peptide feature to address in the matrix
     */
    public void addCorrelationCount(ResiduePositionPair[] domainFeature, ResiduePositionPair[] peptideFeature) {
        int domainIndex = FeatureUtils.getFeatureIndex(domainFeature);
        int peptideIndex = FeatureUtils.getFeatureIndex(peptideFeature);
        if (domainPositionsAsRows) {
            correlationMatrix.add(domainIndex, peptideIndex, 1);
        } else {
            correlationMatrix.add(peptideIndex, domainIndex, 1);
        }
    }

    /**
     * Helper method to access a domain feature from a sparse matrix entry
     * TODO: Remove since not currently used in the code
     */
    protected ResiduePositionPair[] getDomainFeatureFromSparseMatrixEntry(MatrixEntry matrixEntry, ResiduePositionPair[] preAllocatedDomainFeature, int totalDomainSequenceLength) {

        ResiduePositionPair[] domainFeature = null;
        if (correlationMatrix == null)
            return domainFeature;

        int columnIndex = matrixEntry.column();
        int rowIndex = matrixEntry.row();
        if (domainPositionsAsRows) {
            domainFeature = FeatureUtils.indexToFeature(rowIndex, preAllocatedDomainFeature, totalDomainSequenceLength);
        } else {
            domainFeature = FeatureUtils.indexToFeature(columnIndex, preAllocatedDomainFeature, totalDomainSequenceLength);
        }
        return domainFeature;
    }

    /**
     * Helper method to access a peptide feature from a sparse matrix entry
     * TODO: Remove since not currently used in the code
     */
    protected ResiduePositionPair[] getPeptideFeatureFromSparseMatrixEntry(MatrixEntry matrixEntry, ResiduePositionPair[] preAllocatedPeptideFeature, int totalPeptideSequenceLength) {

        ResiduePositionPair[] peptideFeature = null;
        if (correlationMatrix == null)
            return peptideFeature;

        int columnIndex = matrixEntry.column();
        int rowIndex = matrixEntry.row();
        if (domainPositionsAsRows) {
            peptideFeature = FeatureUtils.indexToFeature(columnIndex, preAllocatedPeptideFeature, totalPeptideSequenceLength);
        } else {
            peptideFeature = FeatureUtils.indexToFeature(rowIndex, preAllocatedPeptideFeature, totalPeptideSequenceLength);
        }
        return peptideFeature;
    }

    /**
     * Returns whether or not the domain positions are along the rows of the protein-protein residue correlation matrix
     * @return true if the domain positions are long the rows of the matrix, otherwise false
     */
    public boolean domainPositionsAsRows()
    {
        return this.domainPositionsAsRows;
    }

    /**
     * Returns the protein-protein residue correlation matrix
     * @return FlexCompRowMatrix matrix representing the protein-protein residue correlation matrix 
     */
    public FlexCompRowMatrix getCorrelationMatrix()
    {
        return correlationMatrix;
    }
}
